<?php
namespace Components\Data;

use Components\Data\Contracts\PlanProvider;
use Components\Data\Contracts\ModuleProvider;
use Components\Data\Contracts\NavigationProvider;
use Components\Data\Contracts\ConfigStoreProvider;
use Components\Data\Contracts\NewsletterProvider;
use Components\Data\Contracts\TemplateCouponProvider;
use Components\Data\Contracts\DatasourceProvider;
use Components\Data\Contracts\FormDataProvider;
use Components\Data\Contracts\AnalyticsProvider;
use Components\Data\Mock\MockPlanProvider;
use Components\Data\Mock\MockModuleProvider;
use Components\Data\Mock\MockNavigationProvider;
use Components\Data\Mock\MockConfigStoreProvider;
use Components\Data\Mock\MockNewsletterProvider;
use Components\Data\Mock\MockTemplateCouponProvider;
use Components\Data\Mock\MockDatasourceProvider;
use Components\Data\Mock\MockFormDataProvider;
use Components\Data\Mock\MockAnalyticsProvider;
use Components\Data\Live\LivePlanProvider;
use Components\Data\Live\LiveModuleProvider;
use Components\Data\Live\LiveNavigationProvider;
use Components\Data\Live\LiveConfigStoreProvider;
use Components\Data\Live\LiveNewsletterProvider;
use Components\Data\Live\LiveTemplateCouponProvider;
use Components\Data\Live\LiveDatasourceProvider;
use Components\Data\Live\LiveFormDataProvider;
use Components\Data\Live\LiveAnalyticsProvider;

/**
 * Centralized access to data providers with mode switching.
 * If constant BEEZ_UI_DEV exists and is true => use mock providers.
 */
class DataRegistry
{
    private static ?PlanProvider $plan = null;
    private static ?ModuleProvider $modules = null;
    private static ?NavigationProvider $navigation = null;
    private static ?ConfigStoreProvider $config = null;
    private static ?NewsletterProvider $newsletter = null;
    private static ?TemplateCouponProvider $templateCoupons = null;
    private static ?DatasourceProvider $datasources = null;
    private static ?FormDataProvider $form = null;
    private static ?AnalyticsProvider $analytics = null;

    /**
     * Determine if we are in dev/mock mode.
     */
    public static function isDev(): bool
    {
        // Primary switch: explicit constant
        if (defined('BEEZ_UI_DEV') && BEEZ_UI_DEV === true) {
            return true;
        }
        return false;
    }

    /**
     * Get Plan provider instance respecting mode.
     */
    public static function plan(): PlanProvider
    {
        if (self::$plan === null) {
            self::$plan = self::isDev() ? new MockPlanProvider() : new LivePlanProvider();
        }
        return self::$plan;
    }

    /**
     * Get Module provider instance respecting mode.
     */
    public static function modules(): ModuleProvider
    {
        if (self::$modules === null) {
            self::$modules = self::isDev() ? new MockModuleProvider() : new LiveModuleProvider();
        }
        return self::$modules;
    }

    /**
     * Get Navigation provider instance respecting mode.
     */
    public static function navigation(): NavigationProvider
    {
        if (self::$navigation === null) {
            self::$navigation = self::isDev() ? new MockNavigationProvider() : new LiveNavigationProvider();
        }
        return self::$navigation;
    }

    /**
     * Get Config store provider instance respecting mode.
     */
    public static function config(): ConfigStoreProvider
    {
        if (self::$config === null) {
            self::$config = self::isDev() ? new MockConfigStoreProvider() : new LiveConfigStoreProvider();
        }
        return self::$config;
    }

    /**
     * Get Newsletter provider instance respecting mode.
     */
    public static function newsletter(): NewsletterProvider
    {
        if (self::$newsletter === null) {
            self::$newsletter = self::isDev() ? new MockNewsletterProvider() : new LiveNewsletterProvider();
        }
        return self::$newsletter;
    }

    /**
     * Get Template Coupons provider instance respecting mode.
     */
    public static function templateCoupons(): TemplateCouponProvider
    {
        if (self::$templateCoupons === null) {
            self::$templateCoupons = self::isDev() ? new MockTemplateCouponProvider() : new LiveTemplateCouponProvider();
        }
        return self::$templateCoupons;
    }

    /**
     * Get Datasources provider instance respecting mode.
     */
    public static function datasources(): DatasourceProvider
    {
        if (self::$datasources === null) {
            self::$datasources = self::isDev() ? new MockDatasourceProvider() : new LiveDatasourceProvider();
        }
        return self::$datasources;
    }

    /**
     * Get Form data provider instance respecting mode.
     */
    public static function form(): FormDataProvider
    {
        if (self::$form === null) {
            self::$form = self::isDev() ? new MockFormDataProvider() : new LiveFormDataProvider();
        }
        return self::$form;
    }

    /**
     * Get Analytics provider instance respecting mode.
     */
    public static function analytics(): AnalyticsProvider
    {
        if (self::$analytics === null) {
            self::$analytics = self::isDev() ? new MockAnalyticsProvider() : new LiveAnalyticsProvider();
        }
        return self::$analytics;
    }

    /**
     * Allow resetting cached providers (e.g., if mode changes at runtime).
     */
    public static function reset(): void
    {
        self::$plan = null;
        self::$modules = null;
        self::$navigation = null;
        self::$config = null;
        self::$newsletter = null;
        self::$templateCoupons = null;
        self::$datasources = null;
        self::$form = null;
        self::$analytics = null;
    }
}
