<?php
// components/Drawer.php
namespace Components;

class Drawer extends Base
{
    /**
     * Render the configuration drawer markup.
     *
     * Follows the same pattern as Modal::render() using output buffering.
     * The HTML markup is kept identical to the existing pages/components/config_drawer.php.
     */
    public static function render(array $opts = []): string
    {
        // Support multiple instances via configurable base id (like Modal)
        $drawerId = isset($opts['id']) && $opts['id'] !== '' ? (string)$opts['id'] : 'config-drawer';
        $titleText = isset($opts['title']) ? (string)$opts['title'] : 'Panel title';
        // Optional visual style: 'default' (edge-aligned) | 'floating' (rounded, spaced)
        $style = isset($opts['style']) && $opts['style'] !== '' ? (string)$opts['style'] : 'default';

        // Derived element ids
        $titleId   = $drawerId . '-title';
        $bodyId    = $drawerId . '-body';
        $loadingId = $drawerId . '-loading';

        // Escaped ids/text for HTML
        $drawerIdEsc = self::h($drawerId);
        $titleIdEsc  = self::h($titleId);
        $bodyIdEsc   = self::h($bodyId);
        $loadingIdEsc= self::h($loadingId);
        $titleTextEsc= self::h($titleText);

        // CSS presets
        $panelCssDefault = 'ml-auto block size-full max-w-2xl transform transition duration-500 ease-in-out data-closed:translate-x-full sm:duration-700';
        $drawerCssDefault = 'relative flex h-full flex-col overflow-y-auto bg-white py-6 shadow-xl dark:bg-gray-800 dark:after:absolute dark:after:inset-y-0 dark:after:left-0 dark:after:w-px dark:after:bg-white/10';

        // Optional vertical scroll hint overlays (similar to ItemList horizontal hints)
        $scrollHints = isset($opts['scroll_hints']) ? (bool)$opts['scroll_hints'] : true;

        // Floating variant (optional)
        if ($style === 'floating') {
            // keep width behavior controlled by the panel (max-w-2xl) and allow a sensible min-width
            $panelCssDefault = $panelCssDefault; // unchanged to preserve width handling
            $drawerCssDefault = 'relative m-2 ml-auto flex flex-col overflow-y-auto bg-white py-6 '
                . 'ring ring-black/5 shadow-lg rounded-xl '
                . 'dark:bg-gray-800 dark:ring-gray-700 '
                . 'max-h-[calc(100dvh-1rem)] min-h-[calc(100dvh-1rem)] md:min-w-[25rem]';
        }

        ob_start();
        ?>
<el-dialog>
    <dialog id="<?= $drawerIdEsc; ?>" aria-labelledby="<?= $titleIdEsc; ?>"
            data-style="<?= self::h($style); ?>" class="fixed inset-0 size-auto max-h-none max-w-none overflow-hidden bg-transparent backdrop:bg-transparent">
        <div tabindex="0" class="absolute inset-0 pl-10 focus:outline-none sm:pl-16">
            <el-dialog-panel
                class="<?= self::h($panelCssDefault); ?> relative"
                <?php if ($scrollHints): ?>
                x-data="{
                   canScrollY: false,
                   atTop: true,
                   atBottom: false,
                   update(){
                       const el = this.$refs.scroller || this.$el;
                       const max = Math.max(0, el.scrollHeight - el.clientHeight);
                       this.canScrollY = el.scrollHeight > (el.clientHeight + 2);
                       this.atTop = el.scrollTop <= 1;
                       this.atBottom = el.scrollTop >= (max - 1);
                   },
                   scrollByY(dy){
                       const el = this.$refs.scroller || this.$el;
                       el.scrollBy({ top: dy, behavior: 'smooth' });
                   }
                }"
                x-init="update(); $nextTick(() => update()); setTimeout(() => update(), 60)"
                @resize.window.debounce.120="update()"
                @mouseOver.debounce.120="update()"
                <?php endif; ?>
            >
                <!-- drawer div -->
                <div class="<?= self::h($drawerCssDefault); ?>"
                     <?php if ($scrollHints): ?>
                     x-ref="scroller"
                     @scroll.passive="update()"
                     <?php endif; ?>>
                    <div class="px-4 sm:px-6">
                        <div class="flex items-start justify-between">
                            <h2 id="<?= $titleIdEsc; ?>" class="text-base font-semibold text-gray-900 dark:text-white"><?= $titleTextEsc; ?></h2>
                            <div class="ml-3 flex h-7 items-center">
                                <button type="button" command="close" commandfor="<?= $drawerIdEsc; ?>"
                                        class="relative rounded-md text-gray-400 hover:text-gray-500 focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-brand dark:hover:text-white dark:focus-visible:outline-brand-500">
                                    <span class="absolute -inset-2.5"></span>
                                    <span class="sr-only">Close panel</span>
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5"
                                         data-slot="icon" aria-hidden="true" class="size-6">
                                        <path d="M6 18 18 6M6 6l12 12" stroke-linecap="round" stroke-linejoin="round"/>
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="relative mt-6 flex-1 px-4 sm:px-6">
                        <!-- Loading overlay -->
                        <?= \Components\Base::loaderOverlay($loadingId,'z-10'); ?>
                        <div id="<?= $bodyIdEsc; ?>"></div>
                    </div>
                </div>
                <?php if ($scrollHints): ?>
                <!-- Vertical scroll hint overlays moved outside the drawer div for correct positioning -->
                <div x-cloak x-show="canScrollY && !atTop" x-transition.opacity
                     class="pointer-events-none absolute inset-x-0 top-0 flex justify-center z-20 mr-2 ">
                    <button type="button" @click.stop="scrollByY(-280)"
                            class="pointer-events-auto inline-flex items-center justify-center h-6 w-full bg-secondary/20  rounded-t-xl text-white shadow-sm ring-1 ring-white/20 hover:bg-secondary/30 focus:outline-none">
                        <!-- up chevron -->
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="size-4">
                            <path stroke-linecap="round" stroke-linejoin="round" d="m4.5 15.75 7.5-7.5 7.5 7.5" />
                        </svg>
                    </button>
                </div>
                <div x-cloak x-show="canScrollY && !atBottom" x-transition.opacity
                     class="pointer-events-none absolute inset-x-0 bottom-4 flex justify-center z-20 mr-2">
                    <button type="button" @click.stop="scrollByY(280)"
                            class="pointer-events-auto inline-flex items-center justify-center h-6 w-full bg-secondary/20  rounded-b-xl text-white shadow-sm ring-1 ring-white/20 hover:bg-secondary/30 focus:outline-none">
                        <!-- down chevron -->
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="size-4">
                            <path stroke-linecap="round" stroke-linejoin="round" d="m19.5 8.25-7.5 7.5-7.5-7.5" />
                        </svg>
                    </button>
                </div>
                <?php endif; ?>
            </el-dialog-panel>
        </div>
    </dialog>
</el-dialog>

<script>
(function(){
    try{
        var bodyId = <?= json_encode($bodyId); ?>;
        var loadingId = <?= json_encode($loadingId); ?>;
        if(window.MBLoader && window.MBLoader.bindHtmxOverlay){
            window.MBLoader.bindHtmxOverlay(bodyId, loadingId, 350);
        }
    }catch(e){/* no-op */}
})();
</script>
<?php
        return ob_get_clean();
    }
}
