<?php
// components/RadialProgress.php
namespace Components;

class RadialProgress extends Base {
    /**
     * Render a simple radial progress indicator using SVG.
     *
     * @param int|float $value Percentage value (0-100)
     * @param array $opts Options:
     *  - radius (int): circle radius in px (default 46)
     *  - size (int): svg width/height (default 100)
     *  - strokeWidth (int): stroke width (default 8)
     *  - trackColor (string): background circle color (default #e5e7eb)
     *  - colorClass (string): progress circle color class (default 'text-red-500')
     *  - wrapperClass (string): wrapper div class (default 'relative flex items-center justify-center')
     *  - svgClass (string): svg class (default 'transform -rotate-90')
     *  - showLabel (bool): show percentage label (default true)
     *  - labelClass (string): class for center label (default 'absolute text-black font-bold text-sm')
     */
    public static function render($value, array $opts = []): string {
        $value = max(0, min(100, (float)$value));

        $radius       = isset($opts['radius']) ? (int)$opts['radius'] : 46;
        $size         = isset($opts['size']) ? (int)$opts['size'] : 100;
        $strokeWidth  = isset($opts['strokeWidth']) ? (int)$opts['strokeWidth'] : 8;
        $colorClass   = trim($opts['colorClass'] ?? 'text-red-500');
        $trackColor   = $opts['trackColor'] ?? 'text-gray-200';
        $wrapperClass = trim($opts['wrapperClass'] ?? 'relative flex items-center justify-center');
        $svgClass     = trim($opts['svgClass'] ?? 'transform -rotate-90');
        $showLabel    = array_key_exists('showLabel', $opts) ? (bool)$opts['showLabel'] : true;
        $labelClass   = trim($opts['labelClass'] ?? 'absolute text-black font-bold text-sm');

        $circumference = 2 * M_PI * $radius;
        $offset = $circumference - ($value / 100 * $circumference);

        $radiusEsc = self::h($radius);
        $sizeEsc = self::h($size);
        $strokeEsc = self::h($strokeWidth);
        $circEsc = self::h($circumference);
        $offsetEsc = self::h($offset);
        $trackColorEsc = self::h($trackColor);

        $wrapperClassEsc = self::h($wrapperClass);
        $svgClassEsc = self::h($svgClass);
        $colorClassEsc = self::h($colorClass);
        $labelClassEsc = self::h($labelClass);
        $valueLabel = self::h((string)($value + 0));

        $labelHtml = '';
        if ($showLabel) {
            $labelHtml = "\n        <span class=\"$labelClassEsc\">{$valueLabel}%</span>";
        }

        $html = <<<HTML
<div class="$wrapperClassEsc">
    <!-- data-svgcss="raw" ensures SvgCssOptimizer keeps this inline (dynamic dashoffset) -->
    <svg width="$sizeEsc" height="$sizeEsc" viewBox="0 0 100 100" class="$svgClassEsc" data-svgcss="raw">
        <!-- Background Circle -->
        <circle cx="50" cy="50" r="$radiusEsc" stroke="currentColor"  class="$trackColorEsc" stroke-width="$strokeEsc" fill="none" />
        <!-- Progress Circle -->
        <circle cx="50" cy="50" r="$radiusEsc" stroke="currentColor" stroke-width="$strokeEsc" fill="none" stroke-dasharray="$circEsc" stroke-dashoffset="$offsetEsc" stroke-linecap="round" class="$colorClassEsc transition-all duration-500" />
    </svg>$labelHtml
</div>
HTML;
        return $html;
    }
}
