<?php
namespace Components\Support;

/**
 * Shared UI color palette for components.
 * Provides a Tailwind utility class mapping for common color tokens.
 *
 * Tokens: blue, indigo, orange, yellow, green, gray, purple, red, pink, brand
 * Each token maps to [bg, text, dot].
 */
class Palette
{
    /**
     * Return the palette mapping.
     * @return array<string, array{bg:string, text:string, dot:string, ring:string}>
     */
    public static function map(): array
    {
        return [
            'blue'   => ['bg' => 'bg-blue-100',   'text' => 'text-blue-600',   'dot' => 'bg-blue-500',   'ring' => 'ring-blue-500'],
            'indigo' => ['bg' => 'bg-indigo-100', 'text' => 'text-indigo-600', 'dot' => 'bg-indigo-500', 'ring' => 'ring-indigo-500'],
            'orange' => ['bg' => 'bg-orange-100', 'text' => 'text-orange-600', 'dot' => 'bg-orange-500', 'ring' => 'ring-orange-500'],
            'yellow' => ['bg' => 'bg-yellow-100', 'text' => 'text-yellow-600', 'dot' => 'bg-yellow-500', 'ring' => 'ring-yellow-500'],
            'green'  => ['bg' => 'bg-green-100',  'text' => 'text-green-600',  'dot' => 'bg-green-500',  'ring' => 'ring-green-500'],
            'gray'   => ['bg' => 'bg-gray-100',   'text' => 'text-gray-600',   'dot' => 'bg-gray-500',   'ring' => 'ring-gray-500'],
            'purple' => ['bg' => 'bg-purple-100', 'text' => 'text-purple-600', 'dot' => 'bg-purple-500', 'ring' => 'ring-purple-500'],
            'red'    => ['bg' => 'bg-red-100',    'text' => 'text-red-600',    'dot' => 'bg-red-500',    'ring' => 'ring-red-500'],
            'pink'   => ['bg' => 'bg-pink-100',   'text' => 'text-pink-600',   'dot' => 'bg-pink-500',   'ring' => 'ring-pink-500'],
            'brand'  => ['bg' => 'bg-brand-100',  'text' => 'text-brand-600',  'dot' => 'bg-brand-500',  'ring' => 'ring-brand-500'],
            // Event palette shortcuts
            'event-bounced' => ['bg' => 'bg-fuchsia-100', 'text' => 'text-fuchsia-600', 'dot' => 'bg-fuchsia-500', 'ring' => 'ring-fuchsia-500'],
            'event-sent'    => ['bg' => 'bg-yellow-100',  'text' => 'text-yellow-600',  'dot' => 'bg-yellow-500',  'ring' => 'ring-yellow-500'],
            'event-opened'  => ['bg' => 'bg-blue-100',    'text' => 'text-blue-600',    'dot' => 'bg-blue-500',    'ring' => 'ring-blue-500'],
            'event-clicked' => ['bg' => 'bg-orange-100',  'text' => 'text-orange-600',  'dot' => 'bg-orange-500',  'ring' => 'ring-orange-500'],
            'event-ordered' => ['bg' => 'bg-green-100',   'text' => 'text-green-600',   'dot' => 'bg-green-500',   'ring' => 'ring-green-500'],
        ];
    }

    /**
     * Resolve classes for a token or pass-through utility string.
     * If $value contains utility classes (bg-*, text-*, etc.), it is returned as-is.
     * Otherwise, it is treated as a palette token and returns "bg-* text-*" for that token.
     */
    public static function classes(string $value, string $fallbackToken = 'gray'): string
    {
        $value = trim($value);
        $isUtility = $value !== '' && (preg_match('/\b(bg|text|from|to|via|fill|stroke|ring|border)-/i', $value) || strpos($value, ' ') !== false);
        if ($isUtility) {
            return $value;
        }
        $pal = self::map();
        $token = $value !== '' ? $value : $fallbackToken;
        if (isset($pal[$token])) {
            return $pal[$token]['bg'] . ' ' . $pal[$token]['text'];
        }
        // Fallback to gray
        $g = $pal[$fallbackToken] ?? ['bg' => 'bg-gray-100', 'text' => 'text-gray-600'];
        return $g['bg'] . ' ' . $g['text'];
    }

    /**
     * Get a background class for a small dot indicator corresponding to a token.
     * Falls back to the token bg if no dedicated dot color.
     */
    public static function dotBg(string $token, string $fallbackToken = 'gray'): string
    {
        $pal = self::map();
        if (isset($pal[$token])) {
            return $pal[$token]['dot'] ?: $pal[$token]['bg'];
        }
        return ($pal[$fallbackToken]['dot'] ?? '') ?: ($pal[$fallbackToken]['bg'] ?? 'bg-gray-500');
    }
}
