<?php
// pages/components/audience_lists.php
// Lists all audience (newsletter) lists with actions opening the global modal
use Components\Router;
use Components\Models\Audience;
use Components\Resources\ResourceList;
use Components\Support\Format;

$oob = (int) (mh_get('oob', 0, 'int'));
$outerAttrs = 'id="audience-lists"';
// Enable event-based refresh: when Audience CRUD emits HX-Trigger events,
// this container will re-fetch itself.
// Note: initial load is triggered by pages/audience.php; we only listen here for updates.
try {
    $listUrl = Router::url('audience_lists', ['partial' => 1]);
    // Only refresh the whole list on create/delete; rows handle self-refresh on update
    $trigs = [
        'audience_lists:created from:body',
        'audience_lists:deleted from:body',
    ];
    $outerAttrs .= ' hx-get="' . \Components\Base::h($listUrl) . '"'
                . ' hx-trigger="' . \Components\Base::h(implode(', ', $trigs)) . '"'
                . ' hx-swap="outerHTML"';
} catch (\Throwable $e) { /* ignore */ }
if ($oob) { $outerAttrs .= ' hx-swap-oob="outerHTML"'; }

// Build items using the Audience model (excludes smartlists per model logic)
$items = [];
try {
    $lists = Audience::all();
    foreach ($lists as $list) {
        $id = (int)($list->newsletter_list_id ?? 0);
        $name = (string)($list->newsletter_list_name ?? ('List ' . $id));
        $icon = is_callable([$list, 'icon']) ? $list->icon() : ['svg' => \Components\Base::SVG_OUTLINE_PLAY, 'color' => 'indigo', 'size' => 'sm'];
        // Build a self-refreshing result_count badge (shown on the right next to any status badges)
        $countBadge = null;
        try {
            // Read cached count from the same unified cache used by Audience::getResultCount()
            $ttlSeconds = 900; // 15 minutes
            $rc = (array) $list->getResultCountCache($ttlSeconds);
            $cachedVal = isset($rc['value']) ? $rc['value'] : null;
            $isValid = isset($rc['is_valid']) ? (bool)$rc['is_valid'] : false;

            // Determine label: show cached value (formatted) when present; otherwise an ellipsis
            $label = '…';
            if ($cachedVal !== null) {
                try {
                    $label = Format::humanNumber((int)$cachedVal, [
                        'decimals' => 1,
                        'trim_zeros' => true,
                    ]);
                } catch (\Throwable $e) {
                    $label = (string) (int) $cachedVal;
                }
            }

            // Lazy-load on first view if invalid/stale; keep listening to global refresh events afterwards
            $countBadge = [
                'label_html' => Audience::countAnchorHtml($id, [
                    'intersect' => !$isValid, // trigger lazy fetch on first view when stale
                    'force' => !$isValid,      // ask backend to recompute and persist when stale
                    'label' => $label,
                ]),
                'title' => 'Result count',
                // Use Palette token for color styling
                'color' => 'brand',
                'aria' => 'Result count',
                // Optional: show a small users icon before the count
                'icon' => \Components\Base::SVG_MICRO_USERS,
                'icon_class' => 'size-3! mr-1 -ml-0.5',
            ];
        } catch (\Throwable $e) { $countBadge = null; }
        $items[] = [
            'id' => $id,
            'name' => $name,
            'status' => 'on',
            'icon' => $icon,
            // Rendered by ItemList as badges area (right side). Allow future multiple badges.
            'badges' => array_values(array_filter([$countBadge])),
        ];
    }
} catch (\Throwable $e) { /* ignore */ }

ob_start();

// URLs and lazy-loading behavior are defined within Audience::itemListOptions

echo \Components\Resources\ItemList::render(Audience::itemListOptions([
    'items' => $items,
    'outer_attrs' => $outerAttrs,
    // row_actions can be added here if needed
]));

$__html = ob_get_clean();
echo $__html;
