<?php
namespace SiteZen\Telemetry\Bootstrap;

use SiteZen\Telemetry\Connectors\Database;
use SiteZen\Telemetry\Connectors\Server;
use SiteZen\Telemetry\Connectors\Application;

class Process
{
    private $db;
    private $handlers = [];

    public $request;

    public function __construct(array $config, $request)
    {
        if (!is_array($request) || !isset($request['metric'])) {
            return ["error" => "Unknown request"];
        }

        $this->request = $request;

        $this->db = new Database($config, $request['parameters'] ?? []);

        // Register built-in handlers
        $this->registerDefaultHandlers($config);

    }

    private function registerDefaultHandlers($config): void
    {

        $os_family = $config['PHP_OS_FAMILY'] ?? 'unknown';;
        // Server Metrics
        $this->registerHandler('server:cpu', function () use ($os_family) {
            return Server::cpu($os_family);
        });
        $this->registerHandler('server:memory', function () use ($os_family) {
            return Server::memory($os_family);
        });
        $this->registerHandler('server:storage', function () use ($config) {
            return Server::storage($config);
        });
        $this->registerHandler('server:diskUsage', function () use ($config) {
            return Server::diskUsage($config);
        });
        $this->registerHandler('server:system', function () use ($config, $os_family) {
            return Server::system($config, $os_family);
        });
        $this->registerHandler('server:php', [Server::class, 'php']);
        $this->registerHandler('server:all', function () use ($config, $os_family) {
            return [
                'cpu' => Server::cpu($os_family),
                'memory' => Server::memory($os_family),
                'storage' => Server::storage($config),
                'system' => Server::system($config, $os_family),
                'php' => Server::php(),
            ];
        });

        // Database Metrics
        $this->registerHandler('database:usage', function () {
            return $this->db->usage();
        });
        $this->registerHandler('database:largestTables', function () {
            return $this->db->largestTables();
        });
        $this->registerHandler('database:connections', function () {
            return $this->db->connections();
        });
        $this->registerHandler('database:insight', function () {
            return $this->db->insight();
        });
        $this->registerHandler('database:analyse', function () {
            return $this->db->analyzeProcessList();
        });
        $this->registerHandler('database:processlist', function () {
            return $this->db->processList();
        });
        $this->registerHandler('database:schema', function () {
            return $this->db->schema();
        });
        $this->registerHandler('database:all', function () {
            return [
                'usage' => $this->db->usage(),
                'largestTables' => $this->db->largestTables(),
                'connections' => $this->db->connections(),
                'insight' => $this->db->insight(),
                'analyse' => $this->db->analyzeProcessList(),
                'schema' => $this->db->schema()
            ];
        });
    }

    public function registerHandler(string $metric, callable $callback): void
    {
        $this->handlers[$metric] = $callback;
    }

    public function executeHandler()
    {
        $metric = $this->request['metric']??'unknown';

        if (!isset($this->handlers[$metric])) {
            return ["error" => "Unknown metric"];
        }
        $handler = $this->handlers[$metric];

        // Ensure closure execution
        $data =  is_callable($handler) ? $handler($this->request) : $handler;

        $this->db->close();

        return $data;
    }

}
