<?php
// components/CampaignFlow/AddTail.php
namespace Components\CampaignFlow;

use Components\Base;

class AddTail extends Base
{
    /**
     * Render an end-of-flow "Add step" element.
     * Options:
     * - orientation: 'horizontal'|'vertical' (default: 'horizontal')
     * - size: 'xs'|'sm'|'md'|'lg' (default: 'md')
     * - editable: bool (default: false) If false, renders empty spacer matching connector min size
     * - href: optional link target for the add action
     * - attrs: array of extra attributes (e.g., htmx, data-*). Values are HTML-escaped.
     * - icon: optional SVG/HTML string to use inside the button (defaults to a plus icon)
     */
    public static function render(array $opts = []): string
    {
        $orientation = $opts['orientation'] ?? 'horizontal';
        $size = $opts['size'] ?? 'md';
        $editable = (bool)($opts['editable'] ?? false);
        $href = $opts['href'] ?? null;
        $extraAttrs = is_array($opts['attrs'] ?? null) ? $opts['attrs'] : [];
        $providedIcon = isset($opts['icon']) ? (string)$opts['icon'] : '';

        // Match Connector min spans so the tail integrates seamlessly
        if ($orientation === 'vertical') {
            $minSpan = $size === 'xs' ? 'min-h-[20px]' : ($size === 'sm' ? 'min-h-[20px]' : 'min-h-[40px]');
            $containerCls = 'flex flex-col items-center justify-center';
            $btnWrapCls = 'py-2';
        } else {
            $minSpan = $size === 'xs' ? 'min-w-[3px]' : ($size === 'sm' ? 'min-w-[24px]' : 'min-w-[28px]');
            $containerCls = 'flex items-center justify-center';
            $btnWrapCls = 'px-1.5';
        }

        if (!$editable) {
            $container = self::h($containerCls);
            $min = self::h($minSpan);
            return <<<HTML
<div class="{$container}"><div class="{$min}"></div></div>
HTML;
        }

        // Size map for the button
        $sizes = [
            'xs' => ['btn' => 'size-6', 'icon' => 'size-3.5', 'label' => 'hidden'],
            'sm' => ['btn' => 'size-8', 'icon' => 'size-4', 'label' => 'sr-only'],
            'md' => ['btn' => 'size-10', 'icon' => 'size-5', 'label' => 'text-xs'],
            'lg' => ['btn' => 'size-10', 'icon' => 'size-6', 'label' => 'text-sm'],
        ];
        $sz = $sizes[$size] ?? $sizes['md'];

        $btnBase = 'inline-flex items-center justify-center rounded-full border-2 border-dashed border-flow text-flow hover:text-blue-600 hover:border-blue-400 bg-white shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-300';
        $btnCls = trim($btnBase . ' ' . $sz['btn']);

        // Use provided icon if any; otherwise fallback to default plus icon
        $iconHtml = trim($providedIcon) !== '' ? $providedIcon : '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true"><path d="M12 5v14M5 12h14"/></svg>';
        // Ensure the SVG carries the desired size class for consistent sizing
        if (strpos($iconHtml, '<svg') !== false) {
            if (preg_match('/<svg[^>]*class=\"([^\"]*)\"/i', $iconHtml, $m)) {
                $existing = $m[1];
                $iconHtml = preg_replace('/class=\"[^\"]*\"/i', 'class=\"' . self::h($existing . ' ' . $sz['icon']) . '\"', $iconHtml, 1);
            } else {
                $iconHtml = preg_replace('/<svg\s+/i', '<svg class=\"' . self::h($sz['icon']) . '\" ', $iconHtml, 1);
            }
        }

        $attrPairs = ['type' => 'button', 'title' => 'Add a new step'];
        if ($href) {
            $attrPairs['type'] = null; // remove button type when using link
            $attrPairs['href'] = $href;
        }
        // merge extra attributes
        foreach ($extraAttrs as $k => $v) {
            $attrPairs[$k] = $v;
        }

        $attrs = self::attrs($attrPairs);
        $containerEsc = self::h($containerCls . ' ' . $btnWrapCls);
        $btnClsEsc = self::h($btnCls);

        $tag = $href ? 'a' : 'button';

        return <<<HTML
<div class="$containerEsc">
  <{$tag}{$attrs} class="$btnClsEsc">$iconHtml</{$tag}>
</div>
HTML;
    }
}
