<?php
// components/CampaignFlow/Connector.php
namespace Components\CampaignFlow;

use Components\Base;

class Connector extends Base
{

    const SVG_MINI_CHEVRON_RIGHT = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="8 5 6 10" stroke-width="1" fill="currentColor" class="h-2.5 w-1.5">
    <path fill-rule="evenodd" d="M8.22 5.22a.75.75 0 0 1 1.06 0l4.25 4.25a.75.75 0 0 1 0 1.06l-4.25 4.25a.75.75 0 0 1-1.06-1.06L11.94 10 8.22 6.28a.75.75 0 0 1 0-1.06Z" clip-rule="evenodd" />
</svg>';
    /**
     * Render a connector between steps. If a badge item is provided (wait|condition), shows it centered.
     * Options:
     * - orientation: 'horizontal'|'vertical'
     * - size: 'xs'|'sm'|'md'|'lg'
     */
    public static function render(?array $item, array $opts = []): string
    {
        $orientation = $opts['orientation'] ?? 'horizontal';
        $size = $opts['size'] ?? 'md';
        $editable = (bool)($opts['editable'] ?? false);
        // Optional control over arrow & line presence when rendering a badge connector
        $showLeftArrow = (bool)($opts['left_arrow'] ?? true);
        $showRightArrow = (bool)($opts['right_arrow'] ?? true);
        $showLeftLine  = (bool)($opts['left_line']  ?? true);
        $showRightLine = (bool)($opts['right_line'] ?? true);

        if ($orientation === 'vertical') {
            $lineCls = 'w-px bg-flow';
            $minSpan = $size === 'xs' ? 'min-h-[20px]' : ($size === 'sm' ? 'min-h-[20px]' : 'min-h-[40px]');
            $containerCls = 'flex flex-col items-center justify-center';

            $lineEsc = self::h($lineCls . ' flex-1 ' . $minSpan);
            $containerEsc = self::h($containerCls);

            if ($item) {
                $t = ($item['type'] ?? 'wait');
                if ($t === 'condition') {
                    // Forward special rendering mode if present on item
                    $upperEventLowerCondition = (bool)($item['upper_event_lower_condition'] ?? false);
                    $badgeHtml = ConditionBadge::render($item, [
                        'size' => $size,
                        'orientation' => $orientation,
                        'editable' => $editable,
                        'upper_event_lower_condition' => $upperEventLowerCondition,
                    ]);
                } elseif ($t === 'date_range') {
                    $badgeHtml = DateRangeBadge::render($item, ['size' => $size, 'orientation' => $orientation, 'editable' => $editable]);
                } else {
                    $badgeHtml = WaitBadge::render($item, ['size' => $size, 'orientation' => $orientation, 'editable' => $editable]);
                }
                $html = <<<HTML
<div class="flex px-1.5">
  <div class="$containerEsc">
    <div class="$lineEsc"></div>
    $badgeHtml
    <div class="$lineEsc"></div>
  </div>
</div>
HTML;
                return $html;
            }

            // No wait badge: render a single continuous line (no center gap)
            $html = <<<HTML
<div class="flex px-1.5">
  <div class="$containerEsc">
    <div class="$lineEsc"></div>
  </div>
</div>
HTML;
            return $html;
        }

        // horizontal default
        $lineCls = 'h-px bg-flow ';
        $minSpan = $size === 'xs' ? 'min-w-[3px]' : ($size === 'sm' ? 'min-w-[24px]' : 'min-w-[28px]');
        $containerCls = 'flex flex-row items-center justify-center flex-1';

        $lineEsc = self::h($lineCls . ' flex-1 ' . $minSpan);
        $containerEsc = self::h($containerCls);
        // arrow indicator: same color as the line (via currentColor) and overlap its bounding box
        // slightly position with -ml-3! -mr-1! on the line to eliminate any visible gap from the SVG viewBox padding
        $arrowHtml = '<div class="-ml-1 mr-0.5 shrink-0 leading-none text-flow">' . self::SVG_MINI_CHEVRON_RIGHT . '</div>';
        $leftArrow = $showLeftArrow ? $arrowHtml : '';
        $rightArrow = $showRightArrow ? $arrowHtml : '';

        if ($item) {
            // Prepare optional line segments (left of badge, right of badge)
            $leftLine = $showLeftLine ? '<div class="' . $lineEsc . '"></div>' : '';
            $rightLine = $showRightLine ? '<div class="' . $lineEsc . '"></div>' : '';
            $t = ($item['type'] ?? 'wait');
            if ($t === 'condition') {
                // Forward special rendering mode if present on item
                $upperEventLowerCondition = (bool)($item['upper_event_lower_condition'] ?? false);
                $badgeHtml = ConditionBadge::render($item, [
                    'size' => $size,
                    'orientation' => $orientation,
                    'editable' => $editable,
                    'upper_event_lower_condition' => $upperEventLowerCondition,
                ]);
                // Align behavior with wait badge: place the right arrow at the very end of the right-hand line
                $html = <<<HTML
<div class="$containerEsc">
  $leftLine
  $leftArrow
  $badgeHtml
  $rightLine
  $rightArrow
</div>
HTML;
                return $html;
            } elseif ($t === 'date_range') {
                $badgeHtml = DateRangeBadge::render($item, ['size' => $size, 'orientation' => $orientation, 'editable' => $editable]);
                // Align behavior with wait badge: place the right arrow at the very end of the right-hand line
                $html = <<<HTML
<div class="$containerEsc">
  $leftLine
  $leftArrow
  $badgeHtml
  $rightLine
  $rightArrow
</div>
HTML;
                return $html;
            } else {
                // wait badge: right arrow must be at the very end (after the right-hand line)
                $badgeHtml = WaitBadge::render($item, ['size' => $size, 'orientation' => $orientation, 'editable' => $editable]);
                $html = <<<HTML
<div class="$containerEsc">
  $leftLine
  $leftArrow
  $badgeHtml
  $rightLine
  $rightArrow
</div>
HTML;
                return $html;
            }
        }

        // No wait badge: render a single continuous line (no center spacer)
        $html = <<<HTML
<div class="$containerEsc">
  <div class="$lineEsc"></div>
  $arrowHtml
</div>
HTML;
        return $html;
    }
}
