<?php
// components/CampaignFlow/WaitBadge.php
namespace Components\CampaignFlow;

use Components\Base;

class WaitBadge extends Base
{

    /**
     * Render a wait badge, optionally editable with Alpine and HTMX.
     * $wait keys:
     * - label (string) e.g., "Wait 3 days"
     * - icon (string) e.g., emoji
     * - index (int) optional key for editing
     */
    public static function render(array $wait, array $opts = []): string
    {
        $size = $opts['size'] ?? 'md'; // xs|sm|md|lg
        $orientation = $opts['orientation'] ?? 'horizontal';
        $editable = (bool)($opts['editable'] ?? false);
        $index = isset($wait['index']) ? (int)$wait['index'] : -1;

        $label = (string)($wait['label'] ?? '');
        $labelEsc = self::h($label);
        $icon = (string)($wait['icon'] ?? self::SVG_WAIT);
        $isDisabled = (bool)($wait['disabled'] ?? false);

        // Interactivity options
        $href = isset($wait['href']) ? (string)$wait['href'] : '';
        $extraAttrs = is_array($wait['attrs'] ?? null) ? $wait['attrs'] : [];
        $hasDropdown = !empty($wait['dropdown']) && is_array($wait['dropdown']);

        $sizes = [
            'xs' => ['iconWrap' => 'size-5 p-0.5', 'iconText' => 'text-[10px]', 'label' => 'hidden', 'pad' => 'px-0', 'gap' => 'mx-0'],
            'sm' => ['iconWrap' => 'size-7 p-1', 'iconText' => 'text-xs', 'label' => 'text-[10px] mt-1 whitespace-nowrap', 'pad' => 'px-1', 'gap' => '-mx-0'],
            'md' => ['iconWrap' => 'size-8 p-2', 'iconText' => 'text-sm', 'label' => 'text-xs mt-1 whitespace-nowrap', 'pad' => 'px-2', 'gap' => '-mx-1'],
            'lg' => ['iconWrap' => 'size-9 p-2', 'iconText' => 'text-base', 'label' => 'text-xs mt-1 whitespace-nowrap', 'pad' => 'px-3', 'gap' => 'mx-1'],
        ];
        $sz = $sizes[$size] ?? $sizes['md'];
        $wrapBase = $orientation == 'vertical' ? 'flex flex-row items-center justify-center space-x-1' : 'flex flex-col items-center ' . $sz['gap'] . ' text-secondary shrink-0';

        $iconWrapCls = 'flex items-center justify-center ' . $sz['iconWrap'] . ' rounded-full bg-muted ring ring-flow text-secondary font-bold transition-colors duration-100 group-hover:bg-elevated group-hover:shadow group-hover:ring ring-brand';
        $wrapClsBase = trim($wrapBase . ' ' . $sz['pad'] . ' ' . ($opts['class'] ?? ''));
        if ($isDisabled) {
            $wrapClsBase .= ' ' . self::DISABLED_CLASSES;
        }
        // Mouseover feedback for interactive waits
        $hasAttrs = !empty($wait['attrs']) && is_array($wait['attrs']);
        $hasHref = isset($wait['href']) && (string)$wait['href'] !== '';
        $willHaveDropdown = !empty($wait['dropdown']) && is_array($wait['dropdown']);
        $isInteractive = !$isDisabled && ($hasHref || $hasAttrs || $willHaveDropdown);
        if ($isInteractive) {
            $wrapClsBase .= ' cursor-pointer group';
        }
        $outerClass = trim('relative wait-badge ' . $wrapClsBase);

        $labelPosCls = $orientation == 'vertical' ? '' : 'absolute -bottom-3';

        $iconWrapEsc = self::h($iconWrapCls);
        $labelCls = trim(($labelPosCls) . ' rounded bg-surface px-1 ' . $sz['label'] . ' transition-colors duration-100 group-hover:bg-elevated group-hover:shadow group-hover:ring ring-brand');
        $labelClsEsc = self::h($labelCls);

        // Build the core badge inner HTML
        $badgeInner = <<<HTML
  <div class="$iconWrapEsc" title="$labelEsc">$icon</div>
  <div class="$labelClsEsc">$labelEsc</div>
HTML;

        // If dropdown present, make the badge the dropdown trigger via Components\\Dropdown
        if ($hasDropdown) {
            $dd = $wait['dropdown'];
            $tr = is_array($dd['trigger'] ?? null) ? $dd['trigger'] : [];
            $items = is_array($dd['items'] ?? null) ? $dd['items'] : [];

            $triggerAttrs = array_merge(['class' => $outerClass, 'type' => 'button'], is_array($tr['attrs'] ?? null) ? $tr['attrs'] : []);
            if ($isDisabled) { $triggerAttrs['aria-disabled'] = 'true'; $triggerAttrs['disabled'] = true; }

            return \Components\Dropdown::render([
                'trigger_tag' => 'button',
                'trigger_attrs' => $triggerAttrs,
                'trigger_html' => $badgeInner,
                'items' => $items,
                'anchor' => 'bottom start',
                'popover' => 'manual',
                'wrapper_class' => 'inline-flex relative',
            ]);
        }

        // Otherwise, render as plain div or clickable anchor/button
        $outerTag = ($href !== '') ? 'a' : ((!empty($extraAttrs)) ? 'button' : 'div');
        $outerAttrs = ['class' => $outerClass];
        if ($href !== '' && $outerTag === 'a') { $outerAttrs['href'] = $href; }
        if ($outerTag === 'button') { $outerAttrs['type'] = 'button'; }
        if ($isDisabled) { $outerAttrs['aria-disabled'] = 'true'; if ($outerTag === 'button') { $outerAttrs['disabled'] = true; } }
        foreach ($extraAttrs as $k => $v) { $outerAttrs[$k] = $v; }
        $outerAttrStr = self::attrs($outerAttrs);

        $html = <<<HTML
<{$outerTag}{$outerAttrStr}>
{$badgeInner}
</{$outerTag}>
HTML;
        return $html;
    }
}
