<?php
// components/Common/FieldVisibility.php
namespace Components\Common;

/**
 * Common visibility evaluation used by Form Engine and Assistant Flow.
 * Supports the following visible_if structure on a field/item definition:
 *   visible_if:
 *     field: <controller_field_id>
 *     any_of: ["a", "b"]   # alias: in
 *     eq: "a"
 *     gt: 1
 *     gte: 2
 *
 * Also supports mode-based visibility for create/edit flows via the following flags on a field:
 *   - visible_on: "create" | "edit" | "both" | "view" (default: both)
 *   - edit_only: true (shorthand for visible_on: edit)
 *   - create_only: true (shorthand for visible_on: create)
 *
 * Evaluation uses the submitted/answered values map. For controller fields that are
 * boolean-like (toggle/checkbox), the value is normalized to 'on' when truthy.
 */
class FieldVisibility
{
    /**
     * Quick helper to parse truthy values commonly used in configs.
     */
    protected static function truthy($v): bool
    {
        if ($v === true || $v === 1) return true;
        $s = strtolower(trim((string)$v));
        return in_array($s, ['1','true','on','yes','y'], true);
    }

    /**
     * Determine visibility restrictions based purely on mode (create/edit/view).
     * If mode is empty, returns true (backward compatible).
     */
    public static function isVisibleByMode(array $def, ?string $mode): bool
    {
        $mode = strtolower((string)$mode);
        if ($mode === '') return true;
        if ($mode === 'new' || $mode === 'create') { $mode = 'create'; }
        // Accept 'edit' and 'view' as-is

        $visibleOn = strtolower((string)($def['visible_on'] ?? ''));
        $editOnly = array_key_exists('edit_only', $def) ? self::truthy($def['edit_only']) : false;
        $createOnly = array_key_exists('create_only', $def) ? self::truthy($def['create_only']) : false;

        if ($visibleOn !== '') {
            if ($visibleOn === 'both') return true;
            if ($visibleOn === 'edit') return $mode === 'edit';
            if ($visibleOn === 'create') return $mode === 'create';
            if ($visibleOn === 'view') return $mode === 'view';
            // Unknown value: default allow
            return true;
        }
        if ($editOnly) { return $mode === 'edit'; }
        if ($createOnly) { return $mode === 'create'; }
        return true;
    }

    /**
     * @param array $def Field or item definition that may contain 'visible_if'
     * @param array $submitted Map of submitted/answered values
     * @param array $fieldMeta Optional map of field meta keyed by id (must contain 'type' when available)
     */
    public static function isVisible(array $def, array $submitted, array $fieldMeta = []): bool
    {
        // Mode-based gate first
        $mode = (string)($submitted['__mode'] ?? '');
        if (!self::isVisibleByMode($def, $mode)) return false;

        $cond = $def['visible_if'] ?? null;
        if (!$cond || !is_array($cond)) return true; // no condition -> visible

        $controller = (string)($cond['field'] ?? '');
        if ($controller === '') return true; // invalid rule -> visible
        $raw = $submitted[$controller] ?? '';

        // Try to detect controller type for normalization
        $ctrlDef = $fieldMeta[$controller] ?? [];
        $ctrlType = strtolower((string)($ctrlDef['type'] ?? ''));

        // Normalize checkbox/toggle truthy values to 'on'
        if (in_array($ctrlType, ['toggle', 'checkbox'], true)) {
            $value = ($raw === 'on' || $raw === '1' || $raw === 'true' || $raw === true) ? 'on' : '';
        } else {
            // For arrays (multi-select) we will compare membership for any_of/in; for eq we cast to string
            $value = $raw;
        }

        // any_of / in
        $any = [];
        $anyRaw = null;
        if (isset($cond['any_of']) && is_array($cond['any_of'])) {
            $any = array_map('strval', $cond['any_of']);
            $anyRaw = $cond['any_of'];
        } elseif (isset($cond['in']) && is_array($cond['in'])) {
            $any = array_map('strval', $cond['in']);
            $anyRaw = $cond['in'];
        }
        if (!empty($any)) {
            // Special handling for toggle/checkbox so boolean lists work, e.g. any_of: [true]
            if (in_array($ctrlType, ['toggle', 'checkbox'], true)) {
                $allowed = array_map(function ($x) { return self::truthy($x); }, (array)$anyRaw);
                $valBool = ($value === 'on');
                if (in_array($valBool, $allowed, true)) return true;
            }

            if (is_array($value)) {
                foreach ($value as $v) {
                    if (in_array((string)$v, $any, true)) return true;
                }
            } else {
                if (in_array((string)$value, $any, true)) return true;
            }
        }

        // eq
        if (array_key_exists('eq', $cond)) {
            // Toggle/checkbox: compare as boolean truthiness
            if (in_array($ctrlType, ['toggle', 'checkbox'], true)) {
                $targetBool = self::truthy($cond['eq']);
                $valBool = ($value === 'on');
                if ($valBool === $targetBool) return true;
            }

            if (is_array($value)) {
                if (in_array((string)$cond['eq'], array_map('strval', $value), true)) return true;
            } else {
                if ((string)$value === (string)$cond['eq']) return true;
            }
        }

        // gt/gte numeric comparisons
        $num = null;
        if (!is_array($value) && $value !== '' && is_numeric($value)) $num = (float)$value;
        if ($num !== null) {
            if (isset($cond['gt']) && is_numeric($cond['gt']) && $num > (float)$cond['gt']) return true;
            if (isset($cond['gte']) && is_numeric($cond['gte']) && $num >= (float)$cond['gte']) return true;
        }

        return false; // none matched
    }
}
