<?php
namespace Components\Data\Live;

use Components\Data\Contracts\ConfigStoreProvider;

class LiveConfigStoreProvider implements ConfigStoreProvider
{
    /**
     * In a real implementation, these would proxy to the application backend (DB/API).
     * For now, keep a non-persistent in-memory map for the request lifetime to avoid JSON usage.
     */
    private static array $map = [];

    /**
     * Request-local cache for UI open state map (module => bool)
     */
    private static ?array $uiOpenMap = null;

    public function getAllConfigs(): array
    {
        $out = [];
        // Try to enumerate all known modules via ModuleLoader and read their current values from legacy
        try {
            $all = \Components\ModuleLoader::all();
            foreach ($all as $modKey => $_meta) {
                $out[$modKey] = $this->getModuleConfig((string)$modKey);
            }
        } catch (\Throwable $e) {
            // ignore; fallback to in-memory map
        }
        // Ensure any modules only present in the request-local map are included as well
        foreach (self::$map as $modKey => $vals) {
            if (!isset($out[$modKey])) {
                $out[$modKey] = is_array($vals) ? $vals : [];
            }
        }
        return $out;
    }

    public function getModuleConfig(string $module): array
    {
        $module = (string)$module;
        $out = [];

        // Try to resolve module-specific blueprints (including aliases) and enumerate merged fields
        $blueprintPath = '';
        try {
            // Use merged blueprints (main + alias modules)
            $paths = \Components\ModuleLoader::mergedBlueprintPaths($module);
            if (!is_array($paths) || empty($paths)) {
                $root = defined('BEEZUI_ROOT') ? BEEZUI_ROOT : dirname(__DIR__, 3);
                $blueprintPath = $root . '/blueprint.yaml';
                $engine = new \Components\Form\Engine($blueprintPath);
            } else {
                $engine = \Components\Form\Engine::fromBlueprints($paths);
            }
            $fields = $engine->getFields();

            foreach ($fields as $name => $def) {
                $val = '';
                if (function_exists('mh_cfg')) {
                    try { $val = mh_cfg((string)$name, ''); } catch (\Throwable $e) { $val = ''; }
                }
                $defType = strtolower((string)($def['type'] ?? ''));
                $vType = strtolower((string)($def['validate']['type'] ?? ''));
                if (in_array($defType, ['toggle','checkbox'], true) || $vType === 'boolean') {
                    // Map legacy 'True'/'False' and other truthy values to BeezUI 'on'|''
                    $out[$name] = ($val === 'True' || $val === true || $val === 1 || $val === '1' || $val === 'on' || $val === 'true') ? 'on' : '';
                } else {
                    // Keep raw string (arrays like order_status_multiple will be normalized by renderer)
                    $out[$name] = is_array($val) ? $val : (string)$val;
                }
            }
        } catch (\Throwable $e) {
            // ignore blueprint errors
        }

        // Overlay any request-local overrides captured in the in-memory map for this request
        if (isset(self::$map[$module]) && is_array(self::$map[$module])) {
            $out = array_merge($out, self::$map[$module]);
        }

        return $out;
    }

    public function setModuleConfig(string $module, array $values, bool $processOnlyGivenFields = false): void
    {
        $module = (string)$module;
        // Prepare blueprint fields map for lookup; avoid iterating all fields when not needed
        $fields = null;
        $boolFields = [];
        try {
            $paths = \Components\ModuleLoader::mergedBlueprintPaths($module);
            if (!is_array($paths) || empty($paths)) {
                $root = defined('BEEZUI_ROOT') ? BEEZUI_ROOT : dirname(__DIR__, 3);
                $blueprintPath = $root . '/blueprint.yaml';
                $engine = new \Components\Form\Engine($blueprintPath);
            } else {
                $engine = \Components\Form\Engine::fromBlueprints($paths);
            }
            $fields = $engine->getFields();
            // Build a fast lookup map of boolean fields
            if (is_array($fields)) {
                foreach ($fields as $fname => $def) {
                    $defType = strtolower((string)($def['type'] ?? ''));
                    $vType = strtolower((string)($def['validate']['type'] ?? ''));
                    if (in_array($defType, ['toggle','checkbox'], true) || $vType === 'boolean') {
                        $boolFields[(string)$fname] = true;
                    }
                }
            }
        } catch (\Throwable $e) {
            $fields = null; // fallback to key name heuristics
        }

        // Persist through legacy config where available
        $persistedAny = false;
        // Track which values we actually persisted (used for request-local overlay in partial mode)
        $overlaySubset = [];
        foreach ($values as $key => $val) {
            $cfgKey = (string)$key;
            if ($cfgKey === '') { continue; }
            // Normalize arrays to comma-separated string
            if (is_array($val)) {
                $norm = implode(',', array_map('strval', $val));
            } else {
                $norm = (string)$val;
            }
            // Decide if this key represents a boolean field
            $isBool = (isset($boolFields[$cfgKey]) && $boolFields[$cfgKey] === true) || preg_match('/(_STATUS|_ENABLED?|_ENABLE)$/', $cfgKey);
            // Coerce toggle semantics ONLY for boolean fields: 'on'|'true'|'1' => 'True', '' => 'False'
            if ($isBool) {
                if ($norm === 'on' || $norm === 'true' || $norm === 'True' || $norm === '1') {
                    $norm = 'True';
                } elseif ($norm === '') {
                    $norm = 'False';
                }
            }

            mh_update_config_value([
                'configuration_key' => $cfgKey,
                'configuration_value' => $norm,
            ]);
            $persistedAny = true;
            // Ensure current request sees updated value
            if (function_exists('mh_cfg_override_set')) {
                try { mh_cfg_override_set($cfgKey, $norm); } catch (\Throwable $e2) { /* ignore */ }
            } elseif (function_exists('mh_cfg_refresh')) {
                try { mh_cfg_refresh($cfgKey); } catch (\Throwable $e3) { /* ignore */ }
            }


            // Collect overlay subset for partial updates (use UI-facing value as provided)
            if ($processOnlyGivenFields) {
                $overlaySubset[$cfgKey] = $val;
            }
        }
        // Maintain an in-memory overlay for this request so UI reflects immediately
        if (!isset(self::$map[$module]) || !is_array(self::$map[$module])) {
            self::$map[$module] = [];
        }
        if ($processOnlyGivenFields) {
            // Merge only the actually persisted subset onto the existing overlay
            if (!empty($overlaySubset)) {
                self::$map[$module] = array_merge(self::$map[$module], $overlaySubset);
            }
        } else {
            // Full save: replace overlay entirely with the provided values so UI reflects exact state
            self::$map[$module] = $values;
        }

    }

    public function getAllStatuses(): array
    {
        $out = [];
        try {
            $all = \Components\ModuleLoader::all();
            foreach ($all as $key => $meta) {
                $out[$key] = $this->isModuleOn($key) ? 'on' : '';
            }
        } catch (\Throwable $e) {
            // Fallback: derive from in-memory map using legacy *_STATUS convention
            foreach (self::$map as $module => $vals) {
                if (!is_array($vals)) { $out[$module] = ''; continue; }
                $statusKey = $this->findStatusKey($vals);
                if ($statusKey !== null) {
                    $out[$module] = ($this->isOnValue($vals[$statusKey] ?? '')) ? 'on' : '';
                } else {
                    $out[$module] = '';
                }
            }
        }
        return $out;
    }

    public function isModuleOn(string $module): bool
    {
        $statusKey = $this->resolveStatusKey($module);
        if (is_string($statusKey) && $statusKey !== '' && function_exists('mh_cfg')) {
            try {
                $val = mh_cfg($statusKey);
                // Legacy uses 'True'/'False' strings
                return $val === 'True' || $val === true || $val === 1 || $val === '1';
            } catch (\Throwable $e) {
                // ignore and fallback
            }
        }
        // Fallback to in-memory map status flag
        $vals = $this->getModuleConfig($module);
        if (!empty($vals)) {
            $statusKey = $this->findStatusKey($vals);
            if ($statusKey !== null) {
                return $this->isOnValue($vals[$statusKey] ?? '');
            }
        }
        return false;
    }

    public function setModuleStatus(string $module, bool $on): void
    {
        $statusKey = $this->resolveStatusKey($module);
        if (is_string($statusKey) && $statusKey !== '' && function_exists('mh_update_config_value')) {
            try {
                mh_update_config_value([
                    'configuration_key' => $statusKey,
                    'configuration_value' => $on ? 'True' : 'False',
                ]);
                // Make sure current request sees the updated value without reload
                if (function_exists('mh_cfg_override_set')) {
                    try { mh_cfg_override_set($statusKey, $on ? 'True' : 'False'); } catch (\Throwable $e2) { /* ignore */ }
                } elseif (function_exists('mh_cfg_refresh')) {
                    try { mh_cfg_refresh($statusKey); } catch (\Throwable $e3) { /* ignore */ }
                }
            } catch (\Throwable $e) {
                // ignore and still update in-memory map below
            }
        }
        // Also reflect the change in the in-memory map for UI consistency within this request
        if (!isset(self::$map[$module]) || !is_array(self::$map[$module])) {
            self::$map[$module] = [];
        }
        $keyForMap = $statusKey;
        if (!is_string($keyForMap) || $keyForMap === '') {
            // Fallback: try to discover status key from current config
            $current = $this->getModuleConfig($module);
            $keyForMap = $this->findStatusKey($current) ?? '_STATUS';
        }
        self::$map[$module][$keyForMap] = $on ? 'on' : '';
    }

    public function listKeys(string $module): array
    {
        // Derive from merged blueprints via Form Engine so modules can declare expected keys
        $blueprintPath = '';
        try {
            $paths = \Components\ModuleLoader::mergedBlueprintPaths($module);
            if (!is_array($paths) || empty($paths)) {
                $root = defined('BEEZUI_ROOT') ? BEEZUI_ROOT : dirname(__DIR__, 3);
                $blueprintPath = $root . '/blueprint.yaml';
                $engine = new \Components\Form\Engine($blueprintPath);
            } else {
                $engine = \Components\Form\Engine::fromBlueprints($paths);
            }
            return array_keys($engine->getFields());
        } catch (\Throwable $e) {
            return array_keys(self::$map[$module] ?? []);
        }
    }

    /**
     * Return UI open state map: module => bool
     */
    public function getUiOpenMap(): array
    {
        if (self::$uiOpenMap !== null) {
            return self::$uiOpenMap;
        }
        $map = [];
        // Try to read from legacy configuration key MAILBEEZ_CONFIG_BEEZ_UI_DATA
        if (function_exists('mh_cfg')) {
            try {
                $raw = (string) mh_cfg('MAILBEEZ_CONFIG_BEEZ_UI_DATA', '');
                if ($raw === '') {
                    // Initialize default structured UI data if missing
                    $defaultJson = '{"STATE":{}}';
                    $persisted = false;
                    if (function_exists('mh_insert_or_update_config_value')) {
                        try {
                            mh_insert_or_update_config_value('MAILBEEZ_CONFIG_BEEZ_UI_DATA', $defaultJson);
                            $persisted = true;
                        } catch (\Throwable $e0) { /* ignore */ }
                    }
                    if (!$persisted && function_exists('mh_update_config_value')) {
                        try {
                            mh_update_config_value([
                                'configuration_key' => 'MAILBEEZ_CONFIG_BEEZ_UI_DATA',
                                'configuration_value' => $defaultJson,
                            ]);
                            $persisted = true;
                        } catch (\Throwable $e1) { /* ignore */ }
                    }
                    if ($persisted) {
                        if (function_exists('mh_cfg_override_set')) {
                            try { mh_cfg_override_set('MAILBEEZ_CONFIG_BEEZ_UI_DATA', $defaultJson); } catch (\Throwable $e2) { /* ignore */ }
                        } elseif (function_exists('mh_cfg_refresh')) {
                            try { mh_cfg_refresh('MAILBEEZ_CONFIG_BEEZ_UI_DATA'); } catch (\Throwable $e3) { /* ignore */ }
                        }
                    }
                } else {
                    $decoded = json_decode($raw, true);
                    if (is_array($decoded)) {
                        if (isset($decoded['STATE']) && is_array($decoded['STATE'])) {
                            // New structured format: { "STATE": { module: bool } }
                            foreach ($decoded['STATE'] as $k => $v) {
                                if (is_string($k) && $k !== '') {
                                    $map[$k] = ($v === true || $v === 1 || $v === '1' || $v === 'on' || $v === 'true');
                                }
                            }
                        } else {
                            // Backward compatibility:
                            // - Array of module keys ["modA","modB"]
                            // - Flat object { module: true/false }
                            $isList = array_keys($decoded) === range(0, count($decoded) - 1);
                            if ($isList) {
                                foreach ($decoded as $k) {
                                    if (is_string($k) && $k !== '') { $map[$k] = true; }
                                }
                            } else {
                                foreach ($decoded as $k => $v) {
                                    if (is_string($k) && $k !== '') {
                                        $map[$k] = ($v === true || $v === 1 || $v === '1' || $v === 'on' || $v === 'true');
                                    }
                                }
                            }
                        }
                    }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }
        self::$uiOpenMap = $map;
        return self::$uiOpenMap;
    }

    /**
     * Persist a single module UI open/closed state into MAILBEEZ_CONFIG_BEEZ_UI_DATA using structured format.
     */
    public function setUiOpenState(string $module, bool $open): void
    {
        $module = (string)$module;
        if ($module === '') { return; }

        // Read existing structured data
        $data = [];
        $raw = '';
        if (function_exists('mh_cfg')) {
            try { $raw = (string) mh_cfg('MAILBEEZ_CONFIG_BEEZ_UI_DATA', ''); } catch (\Throwable $e) { $raw = ''; }
        }
        if ($raw !== '') {
            $decoded = json_decode($raw, true);
            if (is_array($decoded)) { $data = $decoded; }
        }
        // Normalize to { STATE: { module: bool } }
        if (!isset($data['STATE']) || !is_array($data['STATE'])) {
            // Migrate from legacy shapes if present
            $state = [];
            if (!empty($data) && is_array($data)) {
                $isList = array_keys($data) === range(0, count($data) - 1);
                if ($isList) {
                    foreach ($data as $k) { if (is_string($k) && $k !== '') { $state[$k] = true; } }
                } else {
                    foreach ($data as $k => $v) { if (is_string($k) && $k !== '') { $state[$k] = ($v === true || $v === 1 || $v === '1' || $v === 'on' || $v === 'true'); } }
                }
            }
            $data = ['STATE' => $state];
        }

        // Ensure the module key exists and update its boolean state
        $data['STATE'][$module] = $open ? true : false;

        // Persist the structured JSON
        $json = json_encode($data, JSON_UNESCAPED_SLASHES);
        if ($json === false) { $json = '{"STATE":{}}'; }
        $persisted = false;
        if (function_exists('mh_insert_or_update_config_value')) {
            try {
                mh_insert_or_update_config_value('MAILBEEZ_CONFIG_BEEZ_UI_DATA', $json);
                $persisted = true;
            } catch (\Throwable $e1) { /* ignore and fallback */ }
        }
        if (!$persisted && function_exists('mh_update_config_value')) {
            try {
                mh_update_config_value([
                    'configuration_key' => 'MAILBEEZ_CONFIG_BEEZ_UI_DATA',
                    'configuration_value' => $json,
                ]);
                $persisted = true;
            } catch (\Throwable $e2) { /* ignore */ }
        }

        // Update request-local cache and runtime cfg
        $cacheMap = [];
        foreach ($data['STATE'] as $k => $v) {
            if (is_string($k) && $k !== '') {
                $cacheMap[$k] = ($v === true || $v === 1 || $v === '1' || $v === 'on' || $v === 'true');
            }
        }
        self::$uiOpenMap = $cacheMap;

        if ($persisted) {
            if (function_exists('mh_cfg_override_set')) {
                try { mh_cfg_override_set('MAILBEEZ_CONFIG_BEEZ_UI_DATA', $json); } catch (\Throwable $e3) { /* ignore */ }
            } elseif (function_exists('mh_cfg_refresh')) {
                try { mh_cfg_refresh('MAILBEEZ_CONFIG_BEEZ_UI_DATA'); } catch (\Throwable $e4) { /* ignore */ }
            }
        }
    }

    private function resolveStatusKey(string $module): ?string
    {
        try {
            $all = \Components\ModuleLoader::all();
            if (isset($all[$module]) && !empty($all[$module]['status_key']) && is_string($all[$module]['status_key'])) {
                return $all[$module]['status_key'];
            }
        } catch (\Throwable $e) {
            // ignore
        }
        // Fallback: inspect in-memory config keys for *_STATUS
        $vals = $this->getModuleConfig($module);
        $found = $this->findStatusKey($vals);
        return $found !== null ? $found : null;
    }

    private function findStatusKey(array $values): ?string
    {
        foreach ($values as $k => $_) {
            if (is_string($k) && preg_match('/_STATUS$/', $k)) {
                return $k;
            }
        }
        return null;
    }

    private function isOnValue($v): bool
    {
        return $v === 'on' || $v === '1' || $v === 1 || $v === true || $v === 'true';
    }
}
