<?php
namespace Components\Data\Live;

use Components\Base;
use Components\Data\Contracts\NavigationProvider;

/**
 * Live provider; replace groups() with real data source (DB/API) when available.
 */
class LiveNavigationProvider implements NavigationProvider
{
    public function groups(): array
    {
        // For now mirror the mock so UI is consistent; swap to real data later.
        return [
            [
                'header' => 'General',
                'items' => [
                    [
                        'key' => 'dashboard',
                        'label' => 'Dashboard',
                        'page' => 'pages/dashboard.php',
                        'submenu' => null,
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
  <path fill-rule="evenodd" d="M3 6a3 3 0 0 1 3-3h2.25a3 3 0 0 1 3 3v2.25a3 3 0 0 1-3 3H6a3 3 0 0 1-3-3V6Zm9.75 0a3 3 0 0 1 3-3H18a3 3 0 0 1 3 3v2.25a3 3 0 0 1-3 3h-2.25a3 3 0 0 1-3-3V6ZM3 15.75a3 3 0 0 1 3-3h2.25a3 3 0 0 1 3 3V18a3 3 0 0 1-3 3H6a3 3 0 0 1-3-3v-2.25Zm9.75 0a3 3 0 0 1 3-3H18a3 3 0 0 1 3 3V18a3 3 0 0 1-3 3h-2.25a3 3 0 0 1-3-3v-2.25Z" clip-rule="evenodd" />
</svg>
'
                    ],
/*
                    [
                        'key' => 'wizard',
                        'label' => 'Setup Wizard',
                        'page' => 'pages/wizard.php',
                        'submenu' => null,
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
  <path fill-rule="evenodd" d="M9 4.5a.75.75 0 0 1 .721.544l.813 2.846a3.75 3.75 0 0 0 2.576 2.576l2.846.813a.75.75 0 0 1 0 1.442l-2.846.813a3.75 3.75 0 0 0-2.576 2.576l-.813 2.846a.75.75 0 0 1-1.442 0l-.813-2.846a3.75 3.75 0 0 0-2.576-2.576l-2.846-.813a.75.75 0 0 1 0-1.442l2.846-.813A3.75 3.75 0 0 0 7.466 7.89l.813-2.846A.75.75 0 0 1 9 4.5ZM18 1.5a.75.75 0 0 1 .728.568l.258 1.036c.236.94.97 1.674 1.91 1.91l1.036.258a.75.75 0 0 1 0 1.456l-1.036.258c-.94.236-1.674.97-1.91 1.91l-.258 1.036a.75.75 0 0 1-1.456 0l-.258-1.036a2.625 2.625 0 0 0-1.91-1.91l-1.036-.258a.75.75 0 0 1 0-1.456l1.036-.258a2.625 2.625 0 0 0 1.91-1.91l.258-1.036A.75.75 0 0 1 18 1.5ZM16.5 15a.75.75 0 0 1 .712.513l.394 1.183c.15.447.5.799.948.948l1.183.395a.75.75 0 0 1 0 1.422l-1.183.395c-.447.15-.799.5-.948.948l-.395 1.183a.75.75 0 0 1-1.422 0l-.395-1.183a1.5 1.5 0 0 0-.948-.948l-1.183-.395a.75.75 0 0 1 0-1.422l1.183-.395c.447-.15.799-.5.948-.948l.395-1.183A.75.75 0 0 1 16.5 15Z" clip-rule="evenodd" />
</svg>
'
                    ],
*/
                ],
            ],
            [
                'header' => 'Campaigns',
                'items' => [
                    [
                        'key' => 'trigger',
                        'label' => 'Trigger Emails',
                        'page' => 'pages/trigger.php',
                        'submenu' => null,
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" class="-rotate-15">
  <path d="M3.478 2.404a.75.75 0 0 0-.926.941l2.432 7.905H13.5a.75.75 0 0 1 0 1.5H4.984l-2.432 7.905a.75.75 0 0 0 .926.94 60.519 60.519 0 0 0 18.445-8.986.75.75 0 0 0 0-1.218A60.517 60.517 0 0 0 3.478 2.404Z" />
</svg>
'
                    ],
                    [
                        'key' => 'newsletter',
                        'label' => 'Newsletter',
                        'page' => 'pages/newsletter.php',
                        'submenu' => null,
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
  <path d="M1.5 8.67v8.58a3 3 0 0 0 3 3h15a3 3 0 0 0 3-3V8.67l-8.928 5.493a3 3 0 0 1-3.144 0L1.5 8.67Z" />
  <path d="M22.5 6.908V6.75a3 3 0 0 0-3-3h-15a3 3 0 0 0-3 3v.158l9.714 5.978a1.5 1.5 0 0 0 1.572 0L22.5 6.908Z" />
</svg>
'
                    ],
                    /*
                    [
                        'key' => 'email_preview',
                        'label' => 'Email Preview',
                        'page' => 'pages/email_preview.php',
                        'submenu' => null,
                        'icon' => Base::SVG_MAIL
                    ],
                    */
                    [
                        'key' => 'audience',
                        'label' => 'Audience',
                        'page' => 'pages/audience.php',
                        'submenu' => null,
                        'icon' => Base::SVG_USERS
                    ],
                    /*
                    [
                        'key' => 'template_coupons',
                        'label' => 'Template Coupons',
                        'page' => 'pages/template_coupons.php',
                        'submenu' => null,
                        'icon' => \Components\Base::SVG_SOLID_VOUCHER
                    ],
                    */
                ],
            ],
            [
                'header' => 'Settings',
                'items' => [
                    [
                        'key' => 'configuration',
                        'label' => 'Configuration',
                        'page' => 'pages/configuration.php',
                        'submenu' => 'pages/submenu/configuration.php',
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
  <path fill-rule="evenodd" d="M11.828 2.25c-.916 0-1.699.663-1.85 1.567l-.091.549a.798.798 0 0 1-.517.608 7.45 7.45 0 0 0-.478.198.798.798 0 0 1-.796-.064l-.453-.324a1.875 1.875 0 0 0-2.416.2l-.243.243a1.875 1.875 0 0 0-.2 2.416l.324.453a.798.798 0 0 1 .064.796 7.448 7.448 0 0 0-.198.478.798.798 0 0 1-.608.517l-.55.092a1.875 1.875 0 0 0-1.566 1.849v.344c0 .916.663 1.699 1.567 1.85l.549.091c.281.047.508.25.608.517.06.162.127.321.198.478a.798.798 0 0 1-.064.796l-.324.453a1.875 1.875 0 0 0 .2 2.416l.243.243c.648.648 1.67.733 2.416.2l.453-.324a.798.798 0 0 1 .796-.064c.157.071.316.137.478.198.267.1.47.327.517.608l.092.55c.15.903.932 1.566 1.849 1.566h.344c.916 0 1.699-.663 1.85-1.567l.091-.549a.798.798 0 0 1 .517-.608 7.52 7.52 0 0 0 .478-.198.798.798 0 0 1 .796.064l.453.324a1.875 1.875 0 0 0 2.416-.2l.243-.243c.648-.648.733-1.67.2-2.416l-.324-.453a.798.798 0 0 1-.064-.796c.071-.157.137-.316.198-.478.1-.267.327-.47.608-.517l.55-.091a1.875 1.875 0 0 0 1.566-1.85v-.344c0-.916-.663-1.699-1.567-1.85l-.549-.091a.798.798 0 0 1-.608-.517 7.507 7.507 0 0 0-.198-.478.798.798 0 0 1 .064-.796l.324-.453a1.875 1.875 0 0 0-.2-2.416l-.243-.243a1.875 1.875 0 0 0-2.416-.2l-.453.324a.798.798 0 0 1-.796.064 7.462 7.462 0 0 0-.478-.198.798.798 0 0 1-.517-.608l-.091-.55a1.875 1.875 0 0 0-1.85-1.566h-.344ZM12 15.75a3.75 3.75 0 0 0 0-7.5 3.75 3.75 0 0 0 0 7.5Z" clip-rule="evenodd" />
</svg>
'
                    ],
                    /*
                    [
                        'key' => 'datasources',
                        'label' => 'Data Sources',
                        'page' => 'pages/datasources.php',
                        'submenu' => null,
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
  <path d="M5.507 4.048A3 3 0 0 1 7.785 3h8.43a3 3 0 0 1 2.278 1.048l1.722 2.008A4.533 4.533 0 0 0 19.5 6h-15c-.243 0-.482.02-.715.056l1.722-2.008Z" />
  <path fill-rule="evenodd" d="M1.5 10.5a3 3 0 0 1 3-3h15a3 3 0 1 1 0 6h-15a3 3 0 0 1-3-3Zm15 0a.75.75 0 1 1-1.5 0 .75.75 0 0 1 1.5 0Zm2.25.75a.75.75 0 1 0 0-1.5.75.75 0 0 0 0 1.5ZM4.5 15a3 3 0 1 0 0 6h15a3 3 0 1 0 0-6h-15Zm11.25 3.75a.75.75 0 1 0 0-1.5.75.75 0 0 0 0 1.5ZM19.5 18a.75.75 0 1 1-1.5 0 .75.75 0 0 1 1.5 0Z" clip-rule="evenodd" />
</svg>
'
                    ],
                    */
                ],
            ],
        ];
    }

    public function keyFromPage(string $page): string
    {
        $page = trim(strtolower(str_replace('\\', '/', ltrim((string)$page))));
        if ($page === '') return '';
        if ($page[0] === '/') $page = ltrim($page, '/');
        // Normalize input forms
        $inputFull = $page;
        $inputBase = $this->basenameNoExt($page);
        $inputShort = $inputBase !== '' ? ($inputBase . '.php') : '';

        $exactMatches = [];
        $shortMatches = [];
        $baseMatches = [];

        foreach ($this->groups() as $group) {
            foreach (($group['items'] ?? []) as $item) {
                $key = isset($item['key']) ? strtolower((string)$item['key']) : '';
                if ($key === '') continue;
                $itemPage = isset($item['page']) ? (string)$item['page'] : '';
                if ($itemPage === '' && $key !== '') {
                    $itemPage = 'pages/' . $key . '.php';
                }
                $itemPage = strtolower(str_replace('\\', '/', ltrim($itemPage, '/')));
                if ($itemPage === '') continue;

                $itemFull = $itemPage;
                $itemBase = $this->basenameNoExt($itemPage);
                $itemShort = $itemBase !== '' ? ($itemBase . '.php') : '';

                if ($inputFull !== '' && $inputFull === $itemFull) {
                    $exactMatches[$key] = true;
                }
                if ($inputShort !== '' && $inputShort === $itemShort) {
                    $shortMatches[$key] = true;
                }
                if ($inputBase !== '' && $inputBase === $itemBase) {
                    $baseMatches[$key] = true;
                }
            }
        }

        if (!empty($exactMatches)) return array_key_first($exactMatches);
        if (!empty($shortMatches)) {
            // If multiple, ambiguous — return '' to be safe
            return count($shortMatches) === 1 ? array_key_first($shortMatches) : '';
        }
        if (!empty($baseMatches)) {
            return count($baseMatches) === 1 ? array_key_first($baseMatches) : '';
        }
        return '';
    }

    public function getTargetId(string $key): string
    {
        $key = trim((string)$key);
        if ($key === '') {
            return '#app-container';
        }
        $norm = strtolower($key);
        // Allow per-item override via 'targetId' or 'target'
        foreach ($this->groups() as $group) {
            foreach (($group['items'] ?? []) as $item) {
                $k = isset($item['key']) ? strtolower((string)$item['key']) : '';
                if ($k === $norm) {
                    $override = $item['targetId'] ?? ($item['target'] ?? null);
                    if (is_string($override) && $override !== '') {
                        $override = (string)$override;
                        return $override[0] === '#' ? $override : ('#' . ltrim($override, '#'));
                    }
                    break 2;
                }
            }
        }
        return '#app-container-' . $norm;
    }

    private function basenameNoExt(string $path): string
    {
        $path = trim($path);
        if ($path === '') return '';
        $path = str_replace('\\', '/', $path);
        $base = basename($path);
        $baseLower = strtolower($base);
        if (substr($baseLower, -4) === '.php') {
            $baseLower = substr($baseLower, 0, -4);
        }
        return $baseLower;
    }
}
