<?php
// components/Form/BaseField.php
namespace Components\Form;

abstract class BaseField extends \Components\Base {
    protected static function helpHtml(?string $help): string {
        if (!$help || trim($help) === '') { return ''; }
        // Delegate to Base::h with limited HTML allowance so this behavior is shared across all outputs
        $allowedTags = '<br><br/><b><strong><i><em><u><a>';
        $safe = \Components\Base::h($help, ['allow_html' => true, 'allowed_tags' => $allowedTags]);
        return <<<HTML
<p class="mt-1 text-sm text-secondary/70 dark:text-secondary">{$safe}</p>
HTML;
    }

    protected static function errorsHtml(array $errors): string {
        if (empty($errors)) return '';
        $errEsc = self::h(implode(' ', $errors));
        return <<<HTML
<p class="mt-1 text-sm text-red-600 dark:text-red-400">{$errEsc}</p>
HTML;
    }

    protected static function fieldBaseClass(bool $hasError): string {
        // Legacy helper retained for components that still use it
        return 'mt-1 block w-full rounded border ' . ($hasError ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-brand');
    }

    /**
     * Map a language code (e.g. en, en_US) to a flag-icons country code (e.g. gb, us)
     */
    protected static function flagCountryFromLang(string $code): ?string {
        $c = strtolower(trim($code));
        if ($c === '') return null;
        // If code contains region (en_US, pt-BR), prefer region part as the country code
        if (strpos($c, '-') !== false || strpos($c, '_') !== false) {
            $parts = preg_split('/[-_]/', $c);
            $region = end($parts);
            if ($region && preg_match('/^[a-z]{2}$/', $region)) return $region;
        }
        // Common language -> country fallbacks
        $map = [
            'en' => 'gb', // default to GB for generic English
            'de' => 'de',
            'fr' => 'fr',
            'es' => 'es',
            'it' => 'it',
            'pt' => 'pt',
            'nl' => 'nl',
            'da' => 'dk',
            'sv' => 'se',
            'fi' => 'fi',
            'no' => 'no',
            'nn' => 'no',
            'nb' => 'no',
            'pl' => 'pl',
            'cs' => 'cz',
            'sk' => 'sk',
            'ro' => 'ro',
            'hu' => 'hu',
            'ru' => 'ru',
            'uk' => 'ua',
            'tr' => 'tr',
            'el' => 'gr',
            'zh' => 'cn',
            'ja' => 'jp',
            'ko' => 'kr',
            'ar' => 'sa',
            'he' => 'il',
            'th' => 'th',
            'vi' => 'vn',
            'id' => 'id',
            'ms' => 'my', // Malay -> Malaysia
        ];
        if (isset($map[$c])) return $map[$c];
        // If code itself is a valid-looking 2-letter country, use it
        if (preg_match('/^[a-z]{2}$/', $c)) return $c;
        return null;
    }

    /**
     * Return span HTML for a flag icon given a language code. Returns empty string if not resolvable.
     */
    protected static function flagIconHtml(?string $langCode, string $extraClasses = ''): string {
        $cc = $langCode ? self::flagCountryFromLang($langCode) : null;
        if (!$cc) return '';
        $cls = 'fi fi-' . self::h($cc) . ($extraClasses ? (' ' . trim($extraClasses)) : '');
        // flag-icons scales with font-size; tailwind text-sm gives a nice size and align a bit with text
        return '<span class="' . $cls . ' text-sm align-[-0.125em]"></span>';
    }
}
