<?php
// components/RadialProgressWidget.php
namespace Components;

class RadialProgressWidget extends Base {
    /**
     * Render the complex radial progress widget (Alpine.js powered) as a reusable component.
     *
     * Options ($opts):
     *  - title (string): label next to the gauge (default 'Performance')
     *  - delta (string|null): right-side delta text, null to hide (default '+25%')
     *  - percent (int|float): percentage value (0-100); if omitted, use first argument
     *  - radius (int): circle radius, must match x-data calculation (default 50)
     *  - cardClass (string): outer card container classes
     *  - gridWrap (bool): whether to include the sample grid/padding wrappers (default false)
     *  - includeAlpine (bool): include Alpine.js v2 script tag (default true)
     *  - colors: trackColorClass (default 'text-gray-300'), progressColorClass (default 'text-blue-600'),
     *            labelColorClass (default 'text-blue-700')
     */
    public static function render($percent = 80, array $opts = []): string {
        $percent = isset($opts['percent']) ? (float)$opts['percent'] : (float)$percent;
        $percent = max(0, min(100, $percent));

        $title  = $opts['title'] ?? 'Performance';
        $delta  = array_key_exists('delta', $opts) ? $opts['delta'] : '+25%';
        $radius = isset($opts['radius']) ? (int)$opts['radius'] : 50;

        $cardClass = trim($opts['cardClass'] ?? 'flex items-center flex-wrap max-w-md px-10 bg-white drop-shadow-xl rounded-2xl h-20');
        $wrapGrid  = isset($opts['gridWrap']) ? (bool)$opts['gridWrap'] : false;
        $includeAlpine = isset($opts['includeAlpine']) ? (bool)$opts['includeAlpine'] : true;

        $trackColorClass   = trim($opts['trackColorClass'] ?? 'text-gray-300');
        $progressColorClass= trim($opts['progressColorClass'] ?? 'text-blue-600');
        $labelColorClass   = trim($opts['labelColorClass'] ?? 'text-blue-700');

        // Escape
        $titleEsc = self::h($title);
        $deltaEsc = $delta !== null ? self::h($delta) : null;
        $cardClassEsc = self::h($cardClass);
        $trackClassEsc = self::h($trackColorClass);
        $progClassEsc  = self::h($progressColorClass);
        $labelClassEsc = self::h($labelColorClass);
        $radiusEsc = self::h($radius);
        $percentJs = json_encode($percent); // safe for inline JS

        $alpine = '';
        if ($includeAlpine) {
            $alpine = "\n<script src=\"https://cdn.jsdelivr.net/gh/alpinejs/alpine@v2.7.3/dist/alpine.min.js\" defer></script>";
        }

        $openWrap = $wrapGrid ? "\n<div class=\"p-12\">\n  <div class=\"grid grid-cols-1 gap-20 lg:grid-cols-2 lg:gap-10\">" : '';
        $closeWrap = $wrapGrid ? "\n  </div>\n</div>" : '';

        $deltaHtml = $deltaEsc !== null ? "\n        <span class=\"ml-auto text-xl font-medium text-blue-600 hidden sm:block\">$deltaEsc</span>" : '';

        $html = <<<HTML
{$alpine}{$openWrap}
    <div class="$cardClassEsc" x-data="{ circumference: {$radiusEsc} * 2 * Math.PI, percent: {$percentJs} }">
        <div class="flex items-center justify-center -m-6 overflow-hidden bg-white rounded-full">
            <svg class="w-32 h-32 transform translate-x-1 translate-y-1" x-cloak aria-hidden="true">
                <circle class="$trackClassEsc" stroke-width="10" stroke="currentColor" fill="transparent" r="{$radiusEsc}" cx="60" cy="60" />
                <circle class="$progClassEsc" stroke-width="10" :stroke-dasharray="circumference" :stroke-dashoffset="circumference - percent / 100 * circumference" stroke-linecap="round" stroke="currentColor" fill="transparent" r="{$radiusEsc}" cx="60" cy="60" />
            </svg>
            <span class="absolute text-2xl $labelClassEsc">{$percentJs}%</span>
        </div>
        <p class="ml-10 font-medium text-gray-600 sm:text-xl">$titleEsc</p>
        {$deltaHtml}
    </div>{$closeWrap}
HTML;
        return $html;
    }
}
