<?php
namespace Components\Resources;

use Components\Base;
use Components\Router;

/**
 * CollectionsList: compact, sidebar-style list of "collections" with counts.
 * Intended for NewsletterCampaign collections, but generic enough for similar use.
 * All inline HTML is captured with ob_start and returned as a string.
 *
 * Options:
 * - items: array of [ id, name, slug, color, count ]
 * - selected_slug: slug of the active collection (string)
 * - page_key: used to determine hx-target container id (defaults to 'newsletter')
 * - new_action: optional array [ label, url, attrs (array), variant ] to open modal/drawer
 * - outer_attrs: string attributes applied to the outer container (e.g., id="..." hx-swap-oob="...")
 */
class CollectionsList
{
    public static function render(array $options): string
    {
        $items = is_array($options['items'] ?? null) ? $options['items'] : [];
        $selected = (string)($options['selected_slug'] ?? '');
        $pageKey = (string)($options['page_key'] ?? 'newsletter');
        $newAction = is_array($options['new_action'] ?? null) ? $options['new_action'] : null;
        $outerAttrs = trim((string)($options['outer_attrs'] ?? ''));

        $targetId = '#app-container-' . $pageKey;

        ob_start();
        ?>
        <aside <?= $outerAttrs; ?> class="w-64 shrink-0">
            <div class="flex items-center justify-between mb-2">
                <h4 class="text-sm font-semibold text-primary">Collections</h4>
                <?php if ($newAction):
                    $attrs = $newAction['attrs'] ?? [];
                    if (!isset($attrs['command'])) { $attrs['command'] = 'show-modal'; }
                    if (!isset($attrs['commandfor'])) { $attrs['commandfor'] = 'config-modal'; }
                    if (!isset($attrs['hx-target'])) { $attrs['hx-target'] = '#config-modal-body'; }
                    if (!isset($attrs['hx-swap'])) { $attrs['hx-swap'] = 'innerHTML'; }
                    if (isset($newAction['url']) && !isset($attrs['hx-get'])) { $attrs['hx-get'] = (string)$newAction['url']; }
                    ?>
                    <button type="button"
                            class="text-xs inline-flex items-center rounded-md bg-primary px-2 py-1 font-medium text-white hover:bg-primary/90"
                        <?php foreach ($attrs as $k=>$v) { echo ' ' . htmlspecialchars((string)$k, ENT_QUOTES, 'UTF-8') . '="' . htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8') . '"'; } ?>>
                        <?= Base::h((string)($newAction['label'] ?? 'New')); ?>
                    </button>
                <?php endif; ?>
            </div>
            <?php if (empty($items)): ?>
                <div class="text-sm text-muted">No collections.</div>
            <?php else: ?>
                <nav class="space-y-1">
                    <?php foreach ($items as $c):
                        $slug = (string)($c['slug'] ?? '');
                        $isActive = ($selected === $slug) || ($selected === '' && $slug === 'all');
                        $full = Router::url('newsletter', $slug !== '' ? ['group' => $slug] : []);
                        $partial = Router::url('newsletter', array_merge(['partial' => 1], $slug !== '' ? ['group' => $slug] : []));
                        $color = (string)($c['color'] ?? '#9ca3af');
                        $name = Base::h((string)($c['name'] ?? 'Untitled'));
                        $count = (int)($c['count'] ?? 0);
                        ?>
                        <a href="<?= $full; ?>"
                           hx-get="<?= $partial; ?>" hx-target="<?= Base::h($targetId); ?>" hx-swap="innerHTML" hx-push-url="<?= $full; ?>"
                           class="flex items-center justify-between rounded px-2 py-2 text-sm <?= $isActive ? 'bg-muted text-brand' : 'text-secondary hover:bg-muted' ?>">
                            <span class="flex items-center gap-2">
                                <span class="inline-block size-2 rounded-full" style="background: <?= Base::h($color); ?>"></span>
                                <?= $name; ?>
                            </span>
                            <span class="text-xs text-secondary"><?= $count; ?></span>
                        </a>
                    <?php endforeach; ?>
                </nav>
            <?php endif; ?>
        </aside>
        <?php
        return ob_get_clean();
    }
}
