<?php
namespace Components\Resources;

use Components\Form\Engine;
use Components\Form\Presentation\ValuePresenter;

class ResourcePresentation
{
    /**
     * Render a generic read-only presentation view for a resource based on Form Engine definitions.
     *
     * Options:
     * - engine: Engine (required)
     * - fields: array of field definitions
     * - groups: array of group definitions
     * - ungrouped: array of ungrouped field names
     * - values: associative array field => current value
     * - container: bool wrap with a space-y container (default true)
     */
    public static function render(array $options): string
    {
        /** @var Engine $engine */
        $engine = $options['engine'] ?? null;
        $fields = $options['fields'] ?? [];
        $groups = $options['groups'] ?? [];
        $ungrouped = $options['ungrouped'] ?? [];
        $values = $options['values'] ?? [];
        $withContainer = array_key_exists('container', $options) ? (bool)$options['container'] : true;
        if (!$engine instanceof Engine) { return ''; }
        ob_start();
        if ($withContainer) { echo '<div class="space-y-4">'; }
        $renderGroupView = function($titleLabel, $names) use ($fields, $engine, $values) {
            if (empty($names)) return;
            $items = [];
            foreach ($names as $name) {
                $def = $fields[$name] ?? null; if (!$def) continue;
                if (!$engine->isFieldPermitted($def)) { continue; }
                // Ensure mode-based visibility applies in presentation view as well
                $valsWithMode = $values;
                if (is_array($valsWithMode)) { $valsWithMode['__mode'] = 'view'; }
                if (!$engine->isFieldVisible($def, $valsWithMode)) { continue; }
                $items[] = $name;
            }
            if (empty($items)) return;
            ob_start();
            ?>
            <div class="">
                <div class="mb-2 text-sm font-semibold text-gray-900 dark:text-gray-100"><?php echo \Components\Base::h((string)$titleLabel); ?></div>
                <div class="space-y-6">
                    <?php foreach ($items as $name): $def = $fields[$name]; $cfgKey = strtoupper($name); $label = \mh_lng($cfgKey . '_TITLE', $def['label'] ?? Engine::labelFromKey($name)); $val = $values[$name] ?? ''; ?>
                        <div class="flex items-start justify-between rounded-md border border-gray-200 bg-gray-50 p-2 dark:border-white/10 dark:bg-white/5">
                            <div>
                                <div class="text-xs font-medium text-gray-700 dark:text-gray-300"><?php echo \Components\Base::h($label); echo Engine::permissionBadge($def); ?></div>
                                <div class="text-sm text-gray-900 dark:text-gray-100"><?php echo ValuePresenter::render($val, $def); ?></div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php
            echo ob_get_clean();
        };
        foreach ($groups as $g) {
            $label = $g['label'] ?? ($g['title'] ?? ($g['id'] ?? ''));
            $names = $g['fields'] ?? [];
            $renderGroupView($label, $names);
        }
        if (!empty($ungrouped)) {
            $renderGroupView(\mh_lng('UI_OTHER','Other'), $ungrouped);
        }
        if ($withContainer) { echo '</div>'; }
        return ob_get_clean();
    }
}
