<?php
// components/Submenu.php
// Reusable submenu renderer

namespace Components;

class Submenu extends Base
{
    /**
     * Render a generic submenu with grouped items using the common module structure.
     * Expected structure: [ groupLabel => [ [ 'key' => string, 'title' => string, ... ], ... ] ]
     * Options:
     * - active: currently active key (default 'overview')
     * - idPrefix: id prefix for anchors (default 'submod-')
     * - hxTarget: HTMX target selector (default '#app-main')
     * - hxSwap: HTMX swap mode (default 'innerHTML')
     * - hxPushBuilder: callable(string $key): string that builds hx-push-url
     * - hxGetBuilder: callable(string $key): string that builds hx-get URL
     * - containerSelector: CSS selector for the submenu container (default '#secondary-column')
     */
    public static function render(array $sections, array $options = []): string
    {
//        dd($sections);

        $active = strtolower((string)($options['active'] ?? 'overview'));
        $idPrefix = (string)($options['idPrefix'] ?? 'submod-');
        $hxTarget = (string)($options['hxTarget'] ?? '#app-main');
        $hxSwap = (string)($options['hxSwap'] ?? 'innerHTML');
        $containerSelector = (string)($options['containerSelector'] ?? '#secondary-column');
        $hxGetBuilder = $options['hxGetBuilder'] ?? null;
        $hxPushBuilder = $options['hxPushBuilder'] ?? null;

        if (!is_callable($hxGetBuilder)) {
            throw new \InvalidArgumentException('Submenu::render requires hxGetBuilder callable');
        }
        if (!is_callable($hxPushBuilder)) {
            throw new \InvalidArgumentException('Submenu::render requires hxPushBuilder callable');
        }

        ob_start();
        ?>
<nav aria-label="Sidebar" class="flex flex-1 flex-col gap-y-7">
  <?php foreach ($sections as  $group => $items): ?>
    <ul role="list" class="-mx-2 space-y-1">

      <?php if (!empty($group) && $group !== 'Overview'): ?>
        <li>
          <div class="submenu_heading mb-2"><?php echo Base::h($group); ?></div>
        </li>
      <?php endif; ?>
      <?php foreach ($items as $item):
        $label   = $item['title'] ?? '';
        $key     = $item['key'] ?? '';
        $current = ($key !== '' && strtolower($key) === $active);
        $hxGet = (string)call_user_func($hxGetBuilder, $key);
        $hxPush = (string)call_user_func($hxPushBuilder, $key);
      ?>
        <li>
          <a id="<?php echo Base::h($idPrefix . $key); ?>" href="#"
             <?php if ($current) { ?>data-active="true"<?php } ?>
             class="group flex gap-x-3 submenu_nav"
             hx-get="<?php echo Base::h($hxGet); ?>"
             hx-target="<?php echo Base::h($hxTarget); ?>"
             hx-swap="<?php echo Base::h($hxSwap); ?>"
             hx-push-url="<?php echo Base::h($hxPush); ?>"
             hx-on:click="setActiveSubmenuGeneric('<?php echo Base::h($idPrefix); ?>','<?php echo Base::h($containerSelector); ?>','<?php echo Base::h($key); ?>')"
          ><?php echo Base::h($label); ?></a>
        </li>
      <?php endforeach; ?>
    </ul>
  <?php endforeach; ?>
</nav>
<script>
(function(){
  function setActiveSubmenuGeneric(idPrefix, containerSelector, key){
    try{
      if(!key) return;
      const container = document.querySelector(containerSelector) || document;
      const links = container.querySelectorAll('a[id^="' + idPrefix.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&') + '"]');
      const target = String(key).toLowerCase();
      links.forEach((link)=>{
        const k = link.id.replace(idPrefix,'');
        if(String(k).toLowerCase() === target){
          link.setAttribute('data-active','true');
        } else {
          link.removeAttribute('data-active');
        }
      });
    }catch(e){/* no-op */}
  }
  function setActiveFromURL(idPrefix, containerSelector, fallback){
    try{
      const params = new URLSearchParams(window.location.search || '');
      const mod = (params.get('module') || '').toLowerCase() || (fallback || 'overview');
      setActiveSubmenuGeneric(idPrefix, containerSelector, mod);
    }catch(e){/* no-op */}
  }
  // Expose setter for hx-on:click
  window.setActiveSubmenuGeneric = setActiveSubmenuGeneric;
  // Sync on history/htmx events
  document.body.addEventListener('htmx:pushedIntoHistory', function(){ setActiveFromURL('<?php echo Base::h($idPrefix); ?>','<?php echo Base::h($containerSelector); ?>','<?php echo Base::h($active); ?>'); });
  document.body.addEventListener('htmx:afterOnLoad', function(){ setActiveFromURL('<?php echo Base::h($idPrefix); ?>','<?php echo Base::h($containerSelector); ?>','<?php echo Base::h($active); ?>'); });
  window.addEventListener('popstate', function(){ setActiveFromURL('<?php echo Base::h($idPrefix); ?>','<?php echo Base::h($containerSelector); ?>','<?php echo Base::h($active); ?>'); });
})();
</script>
<?php
        return ob_get_clean();
    }

    /**
     * Convenience wrapper for the Configuration submenu.
     */
    public static function renderConfiguration(array $sections, string $activeKey = 'overview'): string
    {

        $hxGetBuilder = function(string $key): string {
            // Use partial=1 so Router uses the page param and the entrypoint returns only the inner content
            return \route('module_config', ['module' => $key, 'partial' => 1, 'mode' => ($key != 'overview') ? 'edit' : 'view']);
        };
        $hxPushBuilder = function(string $key): string {
            $tabParam = Router::tabParam();
            return '?' . $tabParam . '=configuration&module=' . rawurlencode($key);
        };


        return self::render($sections, [
            'active' => $activeKey,
            'idPrefix' => 'submod-',
            'hxTarget' => '#app-main-configuration',
            'hxSwap' => 'innerHTML',
            'hxGetBuilder' => $hxGetBuilder,
            'hxPushBuilder' => $hxPushBuilder,
            'containerSelector' => '#secondary-column',
        ]);
    }
}
