You are a senior PHP 8.2 engineer + UX engineer. Generate a NEW, SWITCHABLE admin frontend for my legacy MailBeez plugin using **vanilla PHP**, **Tailwind CSS + Tailwind UI (Plus license)**, **HTMX**, and **Alpine.js**. No Laravel/Blade. Optional templating engine is **Smarty** (configurable), but default to **plain PHP includes**.

## Context

- Current app is legacy PHP without routing. Auth is handled by the host ecommerce app. Permissions + form structure come from module `blueprint.yaml` files (example below).
- Goal: modern, maintainable UI; HTML-first, server-rendered; small JS via HTMX/Alpine; reusable components; forms auto-generated from YAML.
- build frontend elements as smarty components based on the concept of slots if necessary, e.g. Button, Input, Checkbox, Select, Modals, etc.
- Tailwind plus elements: https://tailwindcss.com/plus/ui-blocks/documentation/llms.txt

### Example blueprint.yaml structure (must be supported)
title: coupon_birthday
version: '4.01'
generated: '2018-10-10'
updated: ''
form:
    permissions:
        saas_user: true
        saas_superuser: true
        saas_admin: true
        core_user: true
        core_admin: true
    fields:
        MAILBEEZ_COUPON_BIRTHDAY_STATUS:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: toggle
                required: true
                rule: ''
                min: ''
                max: ''
        MAILBEEZ_COUPON_BIRTHDAY_BEFORE_DAYS:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: integer
                required: true
                rule: ''
                min: 0
                max: 999
        MAILBEEZ_COUPON_BIRTHDAY_PASSED_DAYS_SKIP:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: integer
                required: true
                rule: 'larger_than_MAILBEEZ_COUPON_BIRTHDAY_BEFORE_DAYS'
                min: 0
                max: 999
        MAILBEEZ_COUPON_BIRTHDAY_SENDER:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: 'Email'
                required: true
                rule: ''
                min: ''
                max: ''
        MAILBEEZ_COUPON_BIRTHDAY_SENDER_NAME:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: 'string'
                required: true
                rule: ''
                min: ''
                max: ''
        MAILBEEZ_COUPON_BIRTHDAY_COUPON_ID:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: ''
                required: false
                rule: ''
                min: ''
                max: ''
        MAILBEEZ_COUPON_BIRTHDAY_EXPIRE_DAYS:
            permissions:
                saas_user: false
                saas_superuser: true
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: integer
                required: true
                rule: ''
                min: 1
                max: 999
        MAILBEEZ_COUPON_BIRTHDAY_CODE_LENGTH:
            permissions:
                saas_user: false
                saas_superuser: false
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: 'integer'
                required: true
                rule: ''
                min: 8
                max: 16
        MAILBEEZ_COUPON_BIRTHDAY_LAYOUT:
            permissions:
                saas_user: false
                saas_superuser: false
                saas_admin: true
                core_user: true
                core_admin: true
            type: input
            validate:
                type: ''
                required: ''
                rule: ''
                min: ''
                max: ''


# ...many more keys like in the example

## Hard constraints
- PHP 7.47/8+ . No frameworks.
- use **Smarty**   `.tpl`
- Use **Tailwind CSS** (latest stable) with Tailwind UI Plus patterns. Build via Tailwind CLI; allow CDN fallback for dev.
- Use **HTMX** for server-driven partial updates and **Alpine.js** for local UI state. No other JS frameworks.


## Deliverables (generate all code, ready to run)
1) **File tree** (concise) for `appui/`:
    - `beezui.php` (front controller + router bootstrap)
    - `routes.php` (map path -> controller@action)
    - `controllers/` (DashboardController.php, SettingsController.php, ModulesController.php, AssetsController.php)
    - `views/`
        - `layouts/` (base.php, head.php, topnav.php, sidebar.php, footer.php)
        - `components/` (button.php, card.php, form_field.php, input.php, select.php, checkbox.php, textarea.php, modal.php, tab.php, table.php, alert.php)
        - page views (dashboard.php, settings.php, modules/index.php, modules/edit.php)
    - `domain/` (Permission.php, Blueprint.php, FormBuilder.php)
    - `infra/` (YamlLoader.php [Symfony YAML], Response.php, Router.php, Htmx.php utilities)
    - `assets/css/` (tailwind.input.css, tailwind.output.css)
    - `assets/js/` (alpine.init.js if needed)
    - `config/` (ui.php for flags like USE_NEW_UI, TEMPLATE_ENGINE)
    - `vendor/` if Composer is used for symfony/yaml (optional)

2) **Routing**
    - Simple Router class: supports GET/POST, path params, middleware hooks.
    - `.htaccess` (or nginx rules) to rewrite `/admin/appui/*` to `beezui.php`.
    - Examples:
        - GET `/admin/appui` -> DashboardController@index
        - GET `/admin/appui/modules` -> ModulesController@index
        - GET `/admin/appui/modules/{code}/edit` -> ModulesController@edit
        - POST `/admin/appui/modules/{code}` -> ModulesController@update (HTMX-friendly partial response)

3) **Controllers**
    - Read current user + roles from existing session/environment (assume already authenticated by host app).
    - Enforce permissions using `Permission.php` against blueprint’s `form.permissions` and per-field permissions.
    - Use `YamlLoader` to load `modules/{code}/blueprint.yaml`.
    - Render views with `$this->view('path', $data)` (plain PHP) or Smarty equivalent if enabled.

4) **Views (plain PHP includes)**
    - `layouts/base.php` includes Tailwind CSS (`assets/css/tailwind.output.css`), HTMX, Alpine (CDNs ok), and yields content.
    - Components are **functions** or **includes** with minimal API, e.g.:
        - `components/button.php`: `button($label, $variant='primary', $attrs=[])`
        - `components/input.php`: `input_field($name, $label, $value, $type='text', $required=false, $disabled=false, $help=null)`
        - `components/alert.php`: flash messages.
    - HTMX examples in views:
        - Buttons with `hx-get`/`hx-post`, `hx-target`, `hx-swap`.
        - Forms posting with `hx-post`, returning partials for `#formArea`.

5) **YAML → Form generator**
    - `YamlLoader.php` using `symfony/yaml` (Composer) OR bundled minimal YAML parser if Composer is not available.
    - `Blueprint.php` maps the array to a typed object: form-level permissions, `fields` (name, type, label, defaults, validate, field-level permissions).
    - `FormBuilder.php::render(Blueprint $bp, array $values, Permission $perm): string`
        - For each field, check permission; skip or render disabled if not allowed.
        - Map `type: input|textarea|select|checkbox` → component include.
        - Apply `validate.required|min|max|type` into HTML attributes where possible.
        - Emit consistent Tailwind form layout with labels, help, error slots.
    - Support grouping/sections if keys exist (optional).
    - Provide a tiny `Validation.php` (server-side) that reads `validate.*` and returns field errors; surface in the form (HTMX partial).

6) **Tailwind setup**
    - Tailwind CSS 4
    - `tailwind.input.css` with `@tailwind base; @tailwind components; @tailwind utilities;` (+ optional @layer for `.btn`, etc.).
    - `tailwind.config.js` pointing to `views/**/*.php` (and `templates/**/*.tpl` if Smarty).
    - CLI instructions: `npx tailwindcss -i ./assets/css/tailwind.input.css -o ./assets/css/tailwind.output.css --minify --watch`

7) **Toggle with legacy UI**
    - `config/ui.php`: `return ['USE_NEW_UI' => true, 'TEMPLATE_ENGINE' => 'plain'/* or 'smarty' */];`
    - A helper `ui_switch_link()` in topnav: link to legacy admin; legacy gets a link back to new admin.
    - Guard in legacy entry to redirect to new admin when `USE_NEW_UI` is true (configurable).

8) **Security & UX details**
    - CSRF token helper (store in session; verify on POST).
    - Escape output by default (`htmlspecialchars`) in components. If Smarty, enable auto-escape.
    - HTMX partial detection via `HX-Request` header; controllers can return partials without full layout for snappy UX.
    - Accessibility: labels tied to inputs, focus management for modals, keyboard-close (Alpine).

9) **Output format**
    - First: a compact tree of files.
    - Then: **code blocks per file** with the full, ready-to-run contents (keep files short and focused).
    - Keep explanations minimal; prioritize code.
    - Use **PHP 7.4+** features (strict types, readonly where useful, enums if helpful).
    - No placeholder “TODOs” in core files—provide working minimal implementations.
    - Keep Tailwind classes close to Tailwind UI Plus examples (clean, modern).

10) **Must-include minimal examples**
- Router.php (complete), admin/appui.php (bootstrap), routes.php (map)
- A working Dashboard page using Card + Table components
- Modules: list modules, load a module’s `blueprint.yaml`, render form via `FormBuilder`, POST save with validation, return HTMX partial with success alert
- Components: button.php, input.php, textarea.php, checkbox.php, select.php, modal.php, alert.php, table.php
- YamlLoader + Blueprint + FormBuilder
- Tailwind config/input/output stubs and build note
- .htaccess rewrite (Apache) and nginx config snippet

11) **EMPLATE_ENGINE smarty**
- Add `smarty_bootstrap.php`, composer.json for `smarty/smarty` and `symfony/yaml`
- Show how controllers choose renderer based on config

## Keep it concise:
- Print only the **relevant code** per file (no boilerplate comments).
- Use short, readable Tailwind class lists aligned with Tailwind UI Plus.
- Make the example work end-to-end without external edits.

BEGIN.